<?php

declare(strict_types=1);

namespace Drupal\eca_youtube\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;

/**
 * Get comprehensive channel information from YouTube.
 */
#[Action(
  id: 'eca_youtube_get_channel',
  label: new TranslatableMarkup('YouTube: Get Channel Info'),
  category: new TranslatableMarkup('YouTube'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Retrieves comprehensive channel metadata from YouTube including subscriber count, video count, description, and thumbnails. Accepts channel ID or username.'),
  version_introduced: '1.0.0',
)]
final class GetChannelInfo extends YouTubeActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'channel_input' => '',
      'include_snippet' => '1',
      'include_statistics' => '1',
      'include_content_details' => '1',
      'include_branding_settings' => '',
      'channel_token_name' => '',
    ] + $this->getAuthClientIdDefaultConfig()
      + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);
    $form = $this->addGoogleAuthConfigurationForm($form, $form_state);

    $form['channel_input'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Channel ID or Username'),
      '#description' => $this->t('Enter a YouTube channel ID (e.g., UCBa659QWEk1AI4Tg--mrJ2A) or username (e.g., GoogleDevelopers). Supports token replacement.'),
      '#default_value' => $this->configuration['channel_input'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['include_snippet'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Include Basic Info'),
      '#description' => $this->t('Include title, description, thumbnails, and country. Set to "1", "true", or "yes" to include.'),
      '#default_value' => $this->configuration['include_snippet'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['include_statistics'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Include Statistics'),
      '#description' => $this->t('Include subscriber count, video count, and view count. Set to "1", "true", or "yes" to include.'),
      '#default_value' => $this->configuration['include_statistics'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['include_content_details'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Include Content Details'),
      '#description' => $this->t('Include uploads playlist ID. Set to "1", "true", or "yes" to include.'),
      '#default_value' => $this->configuration['include_content_details'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['include_branding_settings'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Include Branding Settings'),
      '#description' => $this->t('Include banner and keywords. Set to "1", "true", or "yes" to include.'),
      '#default_value' => $this->configuration['include_branding_settings'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['channel_token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Channel Token Name'),
      '#description' => $this->t('Optional token name to store channel data. Accessible as [token_name:title], [token_name:subscriber_count], [token_name:video_count], etc.'),
      '#default_value' => $this->configuration['channel_token_name'],
      '#eca_token_replacement' => TRUE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $channel_input = $this->tokenService->replacePlain($this->configuration['channel_input']);
    $channel_token_name = $this->tokenService->replacePlain($this->configuration['channel_token_name']);

    // Build data parts array based on truthy values
    $data_parts = [];
    if (!empty($this->tokenService->replacePlain($this->configuration['include_snippet']))) {
      $data_parts[] = 'snippet';
    }
    if (!empty($this->tokenService->replacePlain($this->configuration['include_statistics']))) {
      $data_parts[] = 'statistics';
    }
    if (!empty($this->tokenService->replacePlain($this->configuration['include_content_details']))) {
      $data_parts[] = 'contentDetails';
    }
    if (!empty($this->tokenService->replacePlain($this->configuration['include_branding_settings']))) {
      $data_parts[] = 'brandingSettings';
    }

    if (empty($data_parts)) {
      $this->logger->error('YouTube get channel action: At least one data type must be included.');
      return;
    }

    // Validate authentication
    [$auth_type, $client_id] = $this->validateAndParseAuth('get channel');
    if (!$auth_type || !$client_id) {
      return;
    }

    if (empty($channel_input)) {
      $this->logger->error('YouTube get channel action: Missing channel ID or username.');
      return;
    }


    $channel_data = $this->youTubeService->getChannelInfo($auth_type, $client_id, $channel_input, $data_parts);

    if (!$channel_data) {
      $this->logger->error('YouTube get channel action: Failed to retrieve channel @channel_input using @auth_type.', [
        '@channel_input' => $channel_input,
        '@auth_type' => $auth_type,
      ]);
      return;
    }

    // Store channel data in token if requested
    if (!empty($channel_token_name)) {
      $this->tokenService->addTokenData($channel_token_name, $channel_data);
    }

  }

}
