<?php

declare(strict_types=1);

namespace Drupal\eca_youtube\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;

/**
 * Get comprehensive video information from YouTube.
 */
#[Action(
  id: 'eca_youtube_get_video',
  label: new TranslatableMarkup('YouTube: Get Video Info'),
  category: new TranslatableMarkup('YouTube'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Retrieves comprehensive video metadata from YouTube including title, description, thumbnails, statistics, and technical details. Accepts video ID or full YouTube URL.'),
  version_introduced: '1.0.0',
)]
final class GetVideo extends YouTubeActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'video_input' => '',
      'include_snippet' => '1',
      'include_statistics' => '1',
      'include_content_details' => '1',
      'include_status' => '',
      'video_token_name' => '',
    ] + $this->getAuthClientIdDefaultConfig()
      + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);
    $form = $this->addGoogleAuthConfigurationForm($form, $form_state);

    $form['video_input'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Video ID or URL'),
      '#description' => $this->t('Enter a YouTube video ID (e.g., dQw4w9WgXcQ) or full YouTube URL. Supports token replacement.'),
      '#default_value' => $this->configuration['video_input'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['include_snippet'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Include Basic Info'),
      '#description' => $this->t('Include title, description, thumbnails, channel info, and tags. Set to "1", "true", or "yes" to include.'),
      '#default_value' => $this->configuration['include_snippet'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['include_statistics'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Include Statistics'),
      '#description' => $this->t('Include view count, like count, and comment count. Set to "1", "true", or "yes" to include.'),
      '#default_value' => $this->configuration['include_statistics'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['include_content_details'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Include Content Details'),
      '#description' => $this->t('Include duration, definition, and caption availability. Set to "1", "true", or "yes" to include.'),
      '#default_value' => $this->configuration['include_content_details'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['include_status'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Include Status'),
      '#description' => $this->t('Include privacy status and upload status. Set to "1", "true", or "yes" to include.'),
      '#default_value' => $this->configuration['include_status'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['video_token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Video Token Name'),
      '#description' => $this->t('Optional token name to store video data. Accessible as [token_name:title], [token_name:video_url], [token_name:video_id], [token_name:view_count], etc.'),
      '#default_value' => $this->configuration['video_token_name'],
      '#eca_token_replacement' => TRUE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $video_input = $this->tokenService->replacePlain($this->configuration['video_input']);
    $video_token_name = $this->tokenService->replacePlain($this->configuration['video_token_name']);

    // Build data parts array based on truthy values
    $data_parts = [];
    if (!empty($this->tokenService->replacePlain($this->configuration['include_snippet']))) {
      $data_parts[] = 'snippet';
    }
    if (!empty($this->tokenService->replacePlain($this->configuration['include_statistics']))) {
      $data_parts[] = 'statistics';
    }
    if (!empty($this->tokenService->replacePlain($this->configuration['include_content_details']))) {
      $data_parts[] = 'contentDetails';
    }
    if (!empty($this->tokenService->replacePlain($this->configuration['include_status']))) {
      $data_parts[] = 'status';
    }

    if (empty($data_parts)) {
      $this->logger->error('YouTube get video action: At least one data type must be included.');
      return;
    }

    // Validate authentication
    [$auth_type, $client_id] = $this->validateAndParseAuth('get video');
    if (!$auth_type || !$client_id) {
      return;
    }

    // Extract video ID from input
    $video_id = $this->validateAndExtractVideoId($video_input, 'get video');
    if (!$video_id) {
      return;
    }


    $video_data = $this->youTubeService->getVideoInfo($auth_type, $client_id, $video_id, $data_parts);

    if (!$video_data) {
      $this->logger->error('YouTube get video action: Failed to retrieve video @video_id using @auth_type.', [
        '@video_id' => $video_id,
        '@auth_type' => $auth_type,
      ]);
      return;
    }

    // Store video data in token if requested
    if (!empty($video_token_name)) {
      $this->tokenService->addTokenData($video_token_name, $video_data);
    }
  }

}
