<?php

declare(strict_types=1);

namespace Drupal\eca_youtube\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;

/**
 * List available captions for a YouTube video.
 */
#[Action(
  id: 'eca_youtube_list_captions',
  label: new TranslatableMarkup('YouTube: List Captions'),
  category: new TranslatableMarkup('YouTube'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Lists all available caption tracks for a YouTube video. Use this before GetCaption to see available options and get caption IDs.'),
  version_introduced: '1.0.0',
)]
final class ListCaptions extends YouTubeActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'video_input' => '',
      'captions_token_name' => '',
    ] + $this->getAuthClientIdDefaultConfig()
      + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);
    $form = $this->addGoogleAuthConfigurationForm($form, $form_state);

    $form['auth_note'] = [
      '#type' => 'markup',
      '#markup' => '<div class="messages messages--warning">' . $this->t('<strong>Important:</strong> This action requires OAuth2 authentication and only works for videos you own or have permission to access. It will not work for other users\' videos.') . '</div>',
    ];

    $form['video_input'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Video ID or URL'),
      '#description' => $this->t('Enter a YouTube video ID or full YouTube URL for a video you own. Supports token replacement.'),
      '#default_value' => $this->configuration['video_input'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['captions_token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Captions Token Name'),
      '#description' => $this->t('Token name to store caption list. Access as:<br/>
        • <strong>[token_name:captions:0:id]</strong> - First caption ID (for GetCaption)<br/>
        • <strong>[token_name:captions:0:language]</strong> - First caption language<br/>
        • <strong>[token_name:captions:0:name]</strong> - First caption display name<br/>
        • <strong>[token_name:count]</strong> - Total number of captions<br/>
        Use the ID values with the GetCaption action.'),
      '#default_value' => $this->configuration['captions_token_name'],
      '#eca_token_replacement' => TRUE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $video_input = $this->tokenService->replacePlain($this->configuration['video_input']);
    $captions_token_name = $this->tokenService->replacePlain($this->configuration['captions_token_name']);

    // Validate authentication
    [$auth_type, $client_id] = $this->validateAndParseAuth('list captions');
    if (!$auth_type || !$client_id) {
      return;
    }

    // Extract video ID from input
    $video_id = $this->validateAndExtractVideoId($video_input, 'list captions');
    if (!$video_id) {
      return;
    }


    // Get available captions
    $captions = $this->youTubeService->getVideoCaptions($auth_type, $client_id, $video_id);

    if (!$captions || empty($captions)) {
      $this->logger->warning('YouTube list captions action: No captions available for video @video_id.', [
        '@video_id' => $video_id,
      ]);

      // Store empty result
      if (!empty($captions_token_name)) {
        $this->tokenService->addTokenData($captions_token_name, [
          'captions' => [],
          'count' => 0,
          'video_id' => $video_id,
        ]);
      }
      return;
    }

    // Prepare caption data with clear structure
    $caption_data = [
      'captions' => $captions,
      'count' => count($captions),
      'video_id' => $video_id,
    ];

    // Store caption list in token if requested
    if (!empty($captions_token_name)) {
      $this->tokenService->addTokenData($captions_token_name, $caption_data);
    }

  }

}
