<?php

declare(strict_types=1);

namespace Drupal\eca_youtube\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;
use Drupal\eca_youtube\YouTubeService;

/**
 * List videos from a YouTube channel with filtering options.
 */
#[Action(
  id: 'eca_youtube_list_channel_videos',
  label: new TranslatableMarkup('YouTube: List Channel Videos'),
  category: new TranslatableMarkup('YouTube'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Retrieves a list of videos from a YouTube channel with advanced filtering and sorting options. Returns comprehensive video metadata for each video.'),
  version_introduced: '1.0.0',
)]
final class ListChannelVideos extends YouTubeActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'channel_input' => '',
      'include_statistics' => '0',
      'include_content_details' => '0',
      'max_results' => '10',
      'videos_token_name' => '',
    ] + $this->getAuthClientIdDefaultConfig()
      + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);
    $form = $this->addGoogleAuthConfigurationForm($form, $form_state);

    $form['channel_input'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Channel ID or Username'),
      '#description' => $this->t('Enter a YouTube channel ID (e.g., UCBa659QWEk1AI4Tg--mrJ2A) or username (e.g., GoogleDevelopers). Supports token replacement.'),
      '#default_value' => $this->configuration['channel_input'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['include_statistics'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Include Video Statistics'),
      '#description' => $this->t('Include view count, like count, and comment count for each video. Set to "1", "true", or "yes" to include.'),
      '#default_value' => $this->configuration['include_statistics'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['include_content_details'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Include Content Details'),
      '#description' => $this->t('Include video duration, definition, and caption availability. Set to "1", "true", or "yes" to include.'),
      '#default_value' => $this->configuration['include_content_details'],
      '#eca_token_replacement' => TRUE,
    ];


    $form['max_results'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum Results'),
      '#description' => $this->t('Maximum number of videos to return (1-50).'),
      '#default_value' => $this->configuration['max_results'],
      '#min' => 1,
      '#max' => 50,
      '#required' => TRUE,
    ];

    $form['videos_token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Videos Token Name'),
      '#description' => $this->t('Optional token name to store video list. Accessible as [token_name:videos:0:title], [token_name:total_videos], etc.'),
      '#default_value' => $this->configuration['videos_token_name'],
      '#eca_token_replacement' => TRUE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $channel_input = $this->tokenService->replacePlain($this->configuration['channel_input']);
    $include_statistics = !empty($this->tokenService->replacePlain($this->configuration['include_statistics']));
    $include_content_details = !empty($this->tokenService->replacePlain($this->configuration['include_content_details']));
    $max_results = (int) $this->configuration['max_results'];
    $videos_token_name = $this->tokenService->replacePlain($this->configuration['videos_token_name']);

    // Validate authentication
    [$auth_type, $client_id] = $this->validateAndParseAuth('list channel videos');
    if (!$auth_type || !$client_id) {
      return;
    }

    if (empty($channel_input)) {
      $this->logger->error('YouTube list channel videos action: Missing channel ID or username.');
      return;
    }


    // Build parameters for the service method
    $list_params = [
      'include_statistics' => $include_statistics,
      'include_content_details' => $include_content_details,
      'max_results' => $max_results,
    ];

    $videos_data = $this->youTubeService->listChannelVideos($auth_type, $client_id, $channel_input, $list_params);

    if (!$videos_data) {
      $this->logger->error('YouTube list channel videos action: Failed to retrieve videos for channel @channel_input using @auth_type.', [
        '@channel_input' => $channel_input,
        '@auth_type' => $auth_type,
      ]);
      return;
    }

    // Store videos data in token if requested
    if (!empty($videos_token_name)) {
      $this->tokenService->addTokenData($videos_token_name, $videos_data);
    }

  }

}
