<?php

declare(strict_types=1);

namespace Drupal\eca_youtube\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;
use Drupal\eca_youtube\YouTubeService;

/**
 * Search for YouTube videos with advanced filtering.
 */
#[Action(
  id: 'eca_youtube_search_videos',
  label: new TranslatableMarkup('YouTube: Search Videos'),
  category: new TranslatableMarkup('YouTube'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Searches YouTube videos with advanced filtering options including channel, date range, duration, and result ordering. Returns video metadata for matching results.'),
  version_introduced: '1.0.0',
)]
final class SearchVideos extends YouTubeActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'search_query' => '',
      'channel_id' => '',
      'published_after' => '',
      'published_before' => '',
      'order' => YouTubeService::ORDER_RELEVANCE,
      'duration' => YouTubeService::DURATION_ANY,
      'max_results' => '10',
      'search_token_name' => '',
    ] + $this->getAuthClientIdDefaultConfig()
      + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);
    $form = $this->addGoogleAuthConfigurationForm($form, $form_state);

    $form['search_query'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Search Query'),
      '#description' => $this->t('Keywords to search for. Leave empty to search all videos (if channel specified).'),
      '#default_value' => $this->configuration['search_query'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['channel_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Channel ID'),
      '#description' => $this->t('Optional channel ID to limit search to specific channel (e.g., UCBa659QWEk1AI4Tg--mrJ2A).'),
      '#default_value' => $this->configuration['channel_id'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['published_after'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Published After'),
      '#description' => $this->t('Search for videos published after this date. Supports multiple formats: For date range fields use [entity:field_date_range:start_date], for Smart Date fields use [entity:field_smart_date:value], for regular datetime fields use [entity:field_date:date]. Also supports string formats like "2023-01-01", "2023-01-01T00:00:00Z", timestamps, and Drupal field formats.'),
      '#default_value' => $this->configuration['published_after'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['published_before'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Published Before'),
      '#description' => $this->t('Search for videos published before this date. Supports multiple formats: For date range fields use [entity:field_date_range:end_date], for Smart Date fields use [entity:field_smart_date:end_value], for regular datetime fields use [entity:field_date:date]. Also supports string formats like "2024-01-01", "2024-01-01T00:00:00Z", timestamps, and Drupal field formats.'),
      '#default_value' => $this->configuration['published_before'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['order'] = [
      '#type' => 'select',
      '#title' => $this->t('Order Results By'),
      '#description' => $this->t('How to order the search results.'),
      '#default_value' => $this->configuration['order'],
      '#options' => $this->youTubeService->getSearchOrderOptions(),
    ];

    $form['duration'] = [
      '#type' => 'select',
      '#title' => $this->t('Duration Filter'),
      '#description' => $this->t('Filter videos by duration.'),
      '#default_value' => $this->configuration['duration'],
      '#options' => $this->youTubeService->getDurationFilterOptions(),
      '#empty_option' => $this->t('- Any Duration -'),
    ];

    $form['max_results'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum Results'),
      '#description' => $this->t('Maximum number of results to return (1-50).'),
      '#default_value' => $this->configuration['max_results'],
      '#min' => 1,
      '#max' => 50,
      '#required' => TRUE,
    ];

    $form['search_token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Search Results Token Name'),
      '#description' => $this->t('Optional token name to store search results. Accessible as [token_name:results:0:title], [token_name:total_results], etc.'),
      '#default_value' => $this->configuration['search_token_name'],
      '#eca_token_replacement' => TRUE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $search_query = $this->tokenService->replace($this->configuration['search_query'], [], ['clear' => TRUE]);
    $channel_id = $this->tokenService->replacePlain($this->configuration['channel_id']);
    $published_after = $this->tokenService->replacePlain($this->configuration['published_after']);
    $published_before = $this->tokenService->replacePlain($this->configuration['published_before']);
    $order = $this->configuration['order'];
    $duration = $this->configuration['duration'];
    $max_results = (int) $this->configuration['max_results'];
    $search_token_name = $this->tokenService->replacePlain($this->configuration['search_token_name']);

    // Validate authentication
    [$auth_type, $client_id] = $this->validateAndParseAuth('search videos');
    if (!$auth_type || !$client_id) {
      return;
    }

    // Validate search parameters
    if (empty($search_query) && empty($channel_id)) {
      $this->logger->error('YouTube search videos action: Either search query or channel ID must be provided.');
      return;
    }


    // Build search parameters
    $search_params = [
      'query' => $search_query,
      'channel_id' => $channel_id,
      'published_after' => $published_after,
      'published_before' => $published_before,
      'order' => $order,
      'duration' => $duration,
      'max_results' => $max_results,
    ];

    $search_results = $this->youTubeService->searchVideos($auth_type, $client_id, $search_params);

    if (!$search_results) {
      $this->logger->error('YouTube search videos action: Failed to perform search using @auth_type.', [
        '@auth_type' => $auth_type,
      ]);
      return;
    }

    // Store search results in token if requested
    if (!empty($search_token_name)) {
      $this->tokenService->addTokenData($search_token_name, $search_results);
    }

  }

}
