<?php

declare(strict_types=1);

namespace Drupal\eca_youtube\Plugin\Action;

use Drupal\Core\Form\FormStateInterface;
use Drupal\eca_google\Plugin\Action\GoogleApiActionBase;
use Drupal\eca_youtube\YouTubeService;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Base class for YouTube ECA actions.
 */
abstract class YouTubeActionBase extends GoogleApiActionBase {

  /**
   * The YouTube service.
   */
  protected YouTubeService $youTubeService;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->youTubeService = $container->get('eca_youtube.youtube');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state): void {
    parent::validateConfigurationForm($form, $form_state);
    $this->validateYouTubeAccess($form, $form_state);
  }

  /**
   * Validates YouTube API access.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  protected function validateYouTubeAccess(array $form, FormStateInterface $form_state): void {
    $auth_client_id = $form_state->getValue('auth_client_id');
    if (!empty($auth_client_id)) {
      $auth_info = $this->googleApiService->parseAuthClientId($auth_client_id);
      if ($auth_info) {
        [$auth_type, $client_id] = [$auth_info['auth_type'], $auth_info['client_id']];

        if (!$this->youTubeService->validateYouTubeAccess($auth_type, $client_id)) {
          $form_state->setErrorByName('auth_client_id', $this->t('The selected Google API client does not have access to YouTube Data API.'));
        }
      }
    }
  }


  /**
   * Validates video ID format.
   *
   * @param string $video_input
   *   The video ID or URL to validate.
   * @param string $operation
   *   The operation being performed (for error messages).
   *
   * @return string|null
   *   The extracted video ID, or NULL if invalid.
   */
  protected function validateAndExtractVideoId(string $video_input, string $operation): ?string {
    if (empty($video_input)) {
      $this->logger->error('YouTube @operation action: Missing video ID or URL.', [
        '@operation' => $operation,
      ]);
      return NULL;
    }

    $video_id = $this->youTubeService->extractVideoId($video_input);
    if (!$video_id) {
      $this->logger->error('YouTube @operation action: Invalid video ID or URL format: @video_input', [
        '@operation' => $operation,
        '@video_input' => $video_input,
      ]);
      return NULL;
    }

    return $video_id;
  }

}