<?php

declare(strict_types=1);

namespace Drupal\eca_youtube;

use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\eca_google\DateTimeFormattingTrait;
use Drupal\eca_google\GoogleApiService;
use Google\Service\YouTube;

/**
 * Service for YouTube Data API operations.
 */
class YouTubeService {

  use StringTranslationTrait;
  use DateTimeFormattingTrait;

  // Transcript format constants
  public const string TRANSCRIPT_FORMAT_ORIGINAL = '';
  public const string TRANSCRIPT_FORMAT_SRT = 'srt';
  public const string TRANSCRIPT_FORMAT_VTT = 'vtt';
  public const string TRANSCRIPT_FORMAT_TTML = 'ttml';

  // Search order constants
  public const string ORDER_RELEVANCE = 'relevance';
  public const string ORDER_DATE = 'date';
  public const string ORDER_RATING = 'rating';
  public const string ORDER_VIEW_COUNT = 'viewCount';
  public const string ORDER_TITLE = 'title';

  // Duration filter constants
  public const string DURATION_ANY = '';
  public const string DURATION_SHORT = 'short';
  public const string DURATION_MEDIUM = 'medium';
  public const string DURATION_LONG = 'long';


  /**
   * The Google API service.
   */
  protected GoogleApiService $googleApiService;

  /**
   * The logger channel.
   */
  protected LoggerChannelInterface $logger;

  /**
   * Constructs a new YouTubeService object.
   *
   * @param \Drupal\eca_google\GoogleApiService $google_api_service
   *   The Google API service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   */
  public function __construct(GoogleApiService $google_api_service, LoggerChannelFactoryInterface $logger_factory) {
    $this->googleApiService = $google_api_service;
    $this->logger = $logger_factory->get('eca_youtube');
  }

  /**
   * Gets available transcript format options.
   *
   * @return array
   *   Array of transcript format options.
   */
  public function getTranscriptFormatOptions(): array {
    return [
      self::TRANSCRIPT_FORMAT_ORIGINAL => $this->t('Original format'),
      self::TRANSCRIPT_FORMAT_SRT => $this->t('SRT (SubRip Subtitle)'),
      self::TRANSCRIPT_FORMAT_VTT => $this->t('WebVTT'),
      self::TRANSCRIPT_FORMAT_TTML => $this->t('TTML (Timed Text Markup Language)'),
    ];
  }

  /**
   * Gets available search order options.
   *
   * @return array
   *   Array of search order options.
   */
  public function getSearchOrderOptions(): array {
    return [
      self::ORDER_RELEVANCE => $this->t('Relevance'),
      self::ORDER_DATE => $this->t('Upload Date'),
      self::ORDER_RATING => $this->t('Rating'),
      self::ORDER_VIEW_COUNT => $this->t('View Count'),
      self::ORDER_TITLE => $this->t('Title (Alphabetical)'),
    ];
  }

  /**
   * Gets available duration filter options.
   *
   * @return array
   *   Array of duration filter options.
   */
  public function getDurationFilterOptions(): array {
    return [
      self::DURATION_ANY => $this->t('Any Duration'),
      self::DURATION_SHORT => $this->t('Short (< 4 minutes)'),
      self::DURATION_MEDIUM => $this->t('Medium (4-20 minutes)'),
      self::DURATION_LONG => $this->t('Long (> 20 minutes)'),
    ];
  }

  /**
   * Validates YouTube Data API access for the specified client.
   *
   * @param string $auth_type
   *   The authentication type.
   * @param string $client_id
   *   The client ID for authentication.
   *
   * @return bool
   *   TRUE if access is valid, FALSE otherwise.
   */
  public function validateYouTubeAccess(string $auth_type, string $client_id): bool {
    try {
      $service = $this->googleApiService->getService('youtube', $auth_type, $client_id);
      return $service instanceof YouTube;
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to validate YouTube API access: @message', [
        '@message' => $e->getMessage(),
      ]);
      return FALSE;
    }
  }

  /**
   * Gets video information from YouTube.
   *
   * @param string $auth_type
   *   The authentication type.
   * @param string $client_id
   *   The client ID for authentication.
   * @param string $video_id
   *   The YouTube video ID.
   * @param array $parts
   *   The data parts to retrieve.
   *
   * @return array|null
   *   Video data array or null on failure.
   */
  public function getVideoInfo(string $auth_type, string $client_id, string $video_id, array $parts = ['snippet', 'statistics', 'contentDetails']): ?array {
    try {
      $service = $this->googleApiService->getService('youtube', $auth_type, $client_id);
      if (!$service instanceof YouTube) {
        $this->logger->error('Failed to get YouTube service for video info.');
        return NULL;
      }

      $response = $service->videos->listVideos(implode(',', $parts), [
        'id' => $video_id,
      ]);

      if (empty($response->getItems())) {
        $this->logger->warning('No video found for ID: @video_id', [
          '@video_id' => $video_id,
        ]);
        return NULL;
      }

      $video = $response->getItems()[0];
      return $this->formatVideoResult($video);
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to get video info for @video_id: @message', [
        '@video_id' => $video_id,
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Gets video captions list.
   *
   * @param string $auth_type
   *   The authentication type.
   * @param string $client_id
   *   The client ID for authentication.
   * @param string $video_id
   *   The YouTube video ID.
   *
   * @return array|null
   *   Captions list or null on failure.
   */
  public function getVideoCaptions(string $auth_type, string $client_id, string $video_id): ?array {
    try {
      $service = $this->googleApiService->getService('youtube', $auth_type, $client_id);
      if (!$service instanceof YouTube) {
        $this->logger->error('Failed to get YouTube service for captions.');
        return NULL;
      }

      $response = $service->captions->listCaptions('snippet', $video_id);

      $captions = [];
      foreach ($response->getItems() as $caption) {
        $captions[] = [
          'id' => $caption->getId(),
          'language' => $caption->getSnippet()->getLanguage(),
          'name' => $caption->getSnippet()->getName(),
          'status' => $caption->getSnippet()->getStatus(),
          'auto_generated' => $caption->getSnippet()->getTrackKind() === 'asr',
        ];
      }

      return $captions;
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to get captions for video @video_id: @message', [
        '@video_id' => $video_id,
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Downloads video transcript.
   *
   * @param string $auth_type
   *   The authentication type.
   * @param string $client_id
   *   The client ID for authentication.
   * @param string $caption_id
   *   The caption track ID.
   * @param string $format
   *   The transcript format (optional).
   *
   * @return array|null
   *   Transcript data or null on failure.
   */
  public function downloadTranscript(string $auth_type, string $client_id, string $caption_id, string $format = ''): ?array {
    try {
      $service = $this->googleApiService->getService('youtube', $auth_type, $client_id);
      if (!$service instanceof YouTube) {
        $this->logger->error('Failed to get YouTube service for transcript download.');
        return NULL;
      }

      $params = [];
      if (!empty($format)) {
        $params['tfmt'] = $format;
      }

      $transcript = $service->captions->download($caption_id, $params);

      // Get the response body content
      $body = $transcript->getBody();

      // Try to rewind the stream in case it was already read
      if ($body->isSeekable()) {
        $body->rewind();
      }

      $content = $body->getContents();

      return [
        'caption_id' => $caption_id,
        'content' => $content,
        'format' => $format ?: 'original',
      ];
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to download transcript @caption_id: @message', [
        '@caption_id' => $caption_id,
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Extracts video ID from YouTube URL.
   *
   * @param string $url
   *   The YouTube URL.
   *
   * @return string|null
   *   The video ID or null if not found.
   */
  public function extractVideoId(string $url): ?string {
    // Handle various YouTube URL formats
    $patterns = [
      '/(?:youtube\.com\/watch\?v=|youtu\.be\/|youtube\.com\/embed\/)([a-zA-Z0-9_-]{11})/',
      '/youtube\.com\/v\/([a-zA-Z0-9_-]{11})/',
    ];

    foreach ($patterns as $pattern) {
      if (preg_match($pattern, $url, $matches)) {
        return $matches[1];
      }
    }

    // If it's already just a video ID
    if (preg_match('/^[a-zA-Z0-9_-]{11}$/', $url)) {
      return $url;
    }

    return NULL;
  }

  /**
   * Formats video result data into standardized array.
   *
   * @param \Google\Service\YouTube\Video $video
   *   The video object.
   *
   * @return array
   *   Formatted video data.
   */
  protected function formatVideoResult(YouTube\Video $video): array {
    $formatted = [
      'video_id' => $video->getId(),
      'video_url' => 'https://www.youtube.com/watch?v=' . $video->getId(),
    ];

    // Snippet data (basic info)
    if ($snippet = $video->getSnippet()) {
      $formatted['title'] = $snippet->getTitle();
      $formatted['description'] = $snippet->getDescription();
      $formatted['channel_id'] = $snippet->getChannelId();
      $formatted['channel_title'] = $snippet->getChannelTitle();
      $formatted['published_at'] = $snippet->getPublishedAt();
      $formatted['tags'] = $snippet->getTags() ?: [];
      $formatted['category_id'] = $snippet->getCategoryId();
      $formatted['default_language'] = $snippet->getDefaultLanguage();

      // Thumbnails
      if ($thumbnails = $snippet->getThumbnails()) {
        $formatted['thumbnails'] = [];
        foreach (['default', 'medium', 'high', 'standard', 'maxres'] as $size) {
          $method = 'get' . ucfirst($size);
          if (method_exists($thumbnails, $method) && $thumbnails->$method()) {
            $formatted['thumbnails'][$size] = [
              'url' => $thumbnails->$method()->getUrl(),
              'width' => $thumbnails->$method()->getWidth(),
              'height' => $thumbnails->$method()->getHeight(),
            ];
          }
        }
      }
    }

    // Statistics data
    if ($statistics = $video->getStatistics()) {
      $formatted['view_count'] = (int) $statistics->getViewCount();
      $formatted['like_count'] = (int) $statistics->getLikeCount();
      $formatted['comment_count'] = (int) $statistics->getCommentCount();
    }

    // Content details
    if ($contentDetails = $video->getContentDetails()) {
      $formatted['duration'] = $contentDetails->getDuration();
      $formatted['duration_seconds'] = $this->parseIsoDuration($contentDetails->getDuration());
      $formatted['definition'] = $contentDetails->getDefinition();
      $formatted['caption_available'] = $contentDetails->getCaption() === 'true';
    }

    // Status data (privacy, upload status)
    if ($status = $video->getStatus()) {
      $formatted['privacy_status'] = $status->getPrivacyStatus();
      $formatted['upload_status'] = $status->getUploadStatus();
      $formatted['failure_reason'] = $status->getFailureReason();
      $formatted['rejection_reason'] = $status->getRejectionReason();
      $formatted['made_for_kids'] = $status->getMadeForKids();
      $formatted['embeddable'] = $status->getEmbeddable();
      $formatted['license'] = $status->getLicense();
      $formatted['public_stats_viewable'] = $status->getPublicStatsViewable();
    }

    return $formatted;
  }

  /**
   * Searches for YouTube videos.
   *
   * @param string $auth_type
   *   The authentication type.
   * @param string $client_id
   *   The client ID for authentication.
   * @param array $search_params
   *   Search parameters.
   *
   * @return array|null
   *   Search results array or null on failure.
   */
  public function searchVideos(string $auth_type, string $client_id, array $search_params): ?array {
    try {
      $service = $this->googleApiService->getService('youtube', $auth_type, $client_id);
      if (!$service instanceof YouTube) {
        $this->logger->error('Failed to get YouTube service for video search.');
        return NULL;
      }

      $params = [
        'type' => 'video',
        'part' => 'snippet',
      ];

      // Add search parameters
      if (!empty($search_params['query'])) {
        $params['q'] = $search_params['query'];
      }
      if (!empty($search_params['channel_id'])) {
        $params['channelId'] = $search_params['channel_id'];
      }
      if (!empty($search_params['published_after'])) {
        $formatted_date = $this->formatDateTimeForApi($search_params['published_after']);
        if ($formatted_date) {
          $params['publishedAfter'] = $formatted_date;
        }
      }
      if (!empty($search_params['published_before'])) {
        $formatted_date = $this->formatDateTimeForApi($search_params['published_before']);
        if ($formatted_date) {
          $params['publishedBefore'] = $formatted_date;
        }
      }
      if (!empty($search_params['order'])) {
        $params['order'] = $search_params['order'];
      }
      if (!empty($search_params['duration'])) {
        $params['videoDuration'] = $search_params['duration'];
      }
      if (!empty($search_params['max_results'])) {
        $params['maxResults'] = min((int) $search_params['max_results'], 50);
      }

      $response = $service->search->listSearch('snippet', $params);

      $results = [];
      foreach ($response->getItems() as $item) {
        $results[] = $this->formatSearchResult($item);
      }

      return [
        'results' => $results,
        'total_results' => $response->getPageInfo()->getTotalResults(),
        'results_per_page' => $response->getPageInfo()->getResultsPerPage(),
        'next_page_token' => $response->getNextPageToken(),
      ];
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to search YouTube videos: @message', [
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Gets channel information.
   *
   * @param string $auth_type
   *   The authentication type.
   * @param string $client_id
   *   The client ID for authentication.
   * @param string $channel_input
   *   Channel ID or username.
   * @param array $parts
   *   The data parts to retrieve.
   *
   * @return array|null
   *   Channel data array or null on failure.
   */
  public function getChannelInfo(string $auth_type, string $client_id, string $channel_input, array $parts = ['snippet', 'statistics', 'contentDetails']): ?array {
    try {
      $service = $this->googleApiService->getService('youtube', $auth_type, $client_id);
      if (!$service instanceof YouTube) {
        $this->logger->error('Failed to get YouTube service for channel info.');
        return NULL;
      }

      $params = ['part' => implode(',', $parts)];

      // Determine if input is channel ID or username
      if (preg_match('/^UC[a-zA-Z0-9_-]{22}$/', $channel_input)) {
        $params['id'] = $channel_input;
      } else {
        $params['forUsername'] = $channel_input;
      }

      $response = $service->channels->listChannels($params['part'], $params);

      if (empty($response->getItems())) {
        $this->logger->warning('No channel found for input: @input', [
          '@input' => $channel_input,
        ]);
        return NULL;
      }

      $channel = $response->getItems()[0];
      return $this->formatChannelResult($channel);
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to get channel info for @input: @message', [
        '@input' => $channel_input,
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Formats search result data into standardized array.
   *
   * @param \Google\Service\YouTube\SearchResult $result
   *   The search result object.
   *
   * @return array
   *   Formatted search result data.
   */
  protected function formatSearchResult(YouTube\SearchResult $result): array {
    $formatted = [
      'video_id' => $result->getId()->getVideoId(),
      'video_url' => 'https://www.youtube.com/watch?v=' . $result->getId()->getVideoId(),
    ];

    if ($snippet = $result->getSnippet()) {
      $formatted['title'] = $snippet->getTitle();
      $formatted['description'] = $snippet->getDescription();
      $formatted['channel_id'] = $snippet->getChannelId();
      $formatted['channel_title'] = $snippet->getChannelTitle();
      $formatted['published_at'] = $snippet->getPublishedAt();

      // Thumbnails
      if ($thumbnails = $snippet->getThumbnails()) {
        $formatted['thumbnails'] = [];
        foreach (['default', 'medium', 'high'] as $size) {
          $method = 'get' . ucfirst($size);
          if (method_exists($thumbnails, $method) && $thumbnails->$method()) {
            $formatted['thumbnails'][$size] = [
              'url' => $thumbnails->$method()->getUrl(),
              'width' => $thumbnails->$method()->getWidth(),
              'height' => $thumbnails->$method()->getHeight(),
            ];
          }
        }
      }
    }

    return $formatted;
  }

  /**
   * Formats channel result data into standardized array.
   *
   * @param \Google\Service\YouTube\Channel $channel
   *   The channel object.
   *
   * @return array
   *   Formatted channel data.
   */
  protected function formatChannelResult(YouTube\Channel $channel): array {
    $formatted = [
      'channel_id' => $channel->getId(),
      'channel_url' => 'https://www.youtube.com/channel/' . $channel->getId(),
    ];

    // Snippet data (basic info)
    if ($snippet = $channel->getSnippet()) {
      $formatted['title'] = $snippet->getTitle();
      $formatted['description'] = $snippet->getDescription();
      $formatted['published_at'] = $snippet->getPublishedAt();
      $formatted['country'] = $snippet->getCountry();

      // Thumbnails
      if ($thumbnails = $snippet->getThumbnails()) {
        $formatted['thumbnails'] = [];
        foreach (['default', 'medium', 'high'] as $size) {
          $method = 'get' . ucfirst($size);
          if (method_exists($thumbnails, $method) && $thumbnails->$method()) {
            $formatted['thumbnails'][$size] = [
              'url' => $thumbnails->$method()->getUrl(),
              'width' => $thumbnails->$method()->getWidth(),
              'height' => $thumbnails->$method()->getHeight(),
            ];
          }
        }
      }
    }

    // Statistics data
    if ($statistics = $channel->getStatistics()) {
      $formatted['subscriber_count'] = (int) $statistics->getSubscriberCount();
      $formatted['video_count'] = (int) $statistics->getVideoCount();
      $formatted['view_count'] = (int) $statistics->getViewCount();
      $formatted['subscriber_count_hidden'] = $statistics->getHiddenSubscriberCount();
    }

    // Content details
    if ($contentDetails = $channel->getContentDetails()) {
      if ($relatedPlaylists = $contentDetails->getRelatedPlaylists()) {
        $formatted['uploads_playlist_id'] = $relatedPlaylists->getUploads();
      }
    }

    // Branding settings
    if ($brandingSettings = $channel->getBrandingSettings()) {
      if ($channelSettings = $brandingSettings->getChannel()) {
        $formatted['channel_title'] = $channelSettings->getTitle();
        $formatted['channel_description'] = $channelSettings->getDescription();
        $formatted['channel_keywords'] = $channelSettings->getKeywords();
        $formatted['default_language'] = $channelSettings->getDefaultLanguage();
        $formatted['country'] = $channelSettings->getCountry();
      }

      if ($imageSettings = $brandingSettings->getImage()) {
        $formatted['banner_external_url'] = $imageSettings->getBannerExternalUrl();
        $formatted['banner_image_url'] = $imageSettings->getBannerImageUrl();
        $formatted['banner_mobile_image_url'] = $imageSettings->getBannerMobileImageUrl();
        $formatted['banner_tablet_low_image_url'] = $imageSettings->getBannerTabletLowImageUrl();
        $formatted['banner_tablet_image_url'] = $imageSettings->getBannerTabletImageUrl();
        $formatted['banner_tablet_hd_image_url'] = $imageSettings->getBannerTabletHdImageUrl();
        $formatted['banner_tablet_extra_hd_image_url'] = $imageSettings->getBannerTabletExtraHdImageUrl();
        $formatted['banner_mobile_low_image_url'] = $imageSettings->getBannerMobileLowImageUrl();
        $formatted['banner_mobile_medium_hd_image_url'] = $imageSettings->getBannerMobileMediumHdImageUrl();
        $formatted['banner_mobile_hd_image_url'] = $imageSettings->getBannerMobileHdImageUrl();
        $formatted['banner_mobile_extra_hd_image_url'] = $imageSettings->getBannerMobileExtraHdImageUrl();
        $formatted['banner_tv_image_url'] = $imageSettings->getBannerTvImageUrl();
        $formatted['banner_tv_low_image_url'] = $imageSettings->getBannerTvLowImageUrl();
        $formatted['banner_tv_medium_image_url'] = $imageSettings->getBannerTvMediumImageUrl();
        $formatted['banner_tv_high_image_url'] = $imageSettings->getBannerTvHighImageUrl();
      }

      if ($watchSettings = $brandingSettings->getWatch()) {
        $formatted['text_color'] = $watchSettings->getTextColor();
        $formatted['background_color'] = $watchSettings->getBackgroundColor();
        $formatted['featured_playlist_id'] = $watchSettings->getFeaturedPlaylistId();
      }
    }

    return $formatted;
  }

  /**
   * Lists videos from a specific channel.
   *
   * @param string $auth_type
   *   The authentication type.
   * @param string $client_id
   *   The client ID for authentication.
   * @param string $channel_input
   *   Channel ID or username.
   * @param array $list_params
   *   List parameters (max_results, order, published_after, etc.).
   *
   * @return array|null
   *   Channel videos list or null on failure.
   */
  public function listChannelVideos(string $auth_type, string $client_id, string $channel_input, array $list_params = []): ?array {
    try {
      $service = $this->googleApiService->getService('youtube', $auth_type, $client_id);
      if (!$service instanceof YouTube) {
        $this->logger->error('Failed to get YouTube service for channel videos list.');
        return NULL;
      }

      // First get the channel to find the uploads playlist
      $channel_info = $this->getChannelInfo($auth_type, $client_id, $channel_input, ['contentDetails']);

      if (!$channel_info || empty($channel_info['uploads_playlist_id'])) {
        $this->logger->error('Failed to get uploads playlist for channel: @channel', [
          '@channel' => $channel_input,
        ]);
        return NULL;
      }

      $uploads_playlist_id = $channel_info['uploads_playlist_id'];

      // Build playlist items list parameters
      $params = [
        'part' => 'snippet,contentDetails',
        'playlistId' => $uploads_playlist_id,
      ];

      if (!empty($list_params['max_results'])) {
        $params['maxResults'] = min((int) $list_params['max_results'], 50);
      }

      if (!empty($list_params['page_token'])) {
        $params['pageToken'] = $list_params['page_token'];
      }

      // Get playlist items (which are the channel's videos)
      $response = $service->playlistItems->listPlaylistItems($params['part'], $params);

      $videos = [];
      $video_ids = [];

      // Collect video IDs for batch video info retrieval
      foreach ($response->getItems() as $item) {
        $video_id = $item->getContentDetails()->getVideoId();
        $video_ids[] = $video_id;

        // Basic info from playlist item
        $videos[$video_id] = [
          'video_id' => $video_id,
          'video_url' => 'https://www.youtube.com/watch?v=' . $video_id,
          'title' => $item->getSnippet()->getTitle(),
          'description' => $item->getSnippet()->getDescription(),
          'published_at' => $item->getSnippet()->getPublishedAt(),
          'position' => $item->getSnippet()->getPosition(),
        ];

        // Add thumbnails
        if ($thumbnails = $item->getSnippet()->getThumbnails()) {
          $videos[$video_id]['thumbnails'] = [];
          foreach (['default', 'medium', 'high'] as $size) {
            $method = 'get' . ucfirst($size);
            if (method_exists($thumbnails, $method) && $thumbnails->$method()) {
              $videos[$video_id]['thumbnails'][$size] = [
                'url' => $thumbnails->$method()->getUrl(),
                'width' => $thumbnails->$method()->getWidth(),
                'height' => $thumbnails->$method()->getHeight(),
              ];
            }
          }
        }
      }

      // Optionally get additional video details (statistics, duration, etc.)
      if (!empty($list_params['include_statistics']) && !empty($video_ids)) {
        $video_details = $service->videos->listVideos('statistics,contentDetails', [
          'id' => implode(',', array_slice($video_ids, 0, 50)), // API limit
        ]);

        foreach ($video_details->getItems() as $video_detail) {
          $video_id = $video_detail->getId();
          if (isset($videos[$video_id])) {
            // Add statistics
            if ($statistics = $video_detail->getStatistics()) {
              $videos[$video_id]['view_count'] = (int) $statistics->getViewCount();
              $videos[$video_id]['like_count'] = (int) $statistics->getLikeCount();
              $videos[$video_id]['comment_count'] = (int) $statistics->getCommentCount();
            }

            // Add content details
            if ($contentDetails = $video_detail->getContentDetails()) {
              $videos[$video_id]['duration'] = $contentDetails->getDuration();
              $videos[$video_id]['duration_seconds'] = $this->parseIsoDuration($contentDetails->getDuration());
            }
          }
        }
      }

      // Convert to indexed array
      $videos_list = array_values($videos);

      return [
        'videos' => $videos_list,
        'total_videos' => count($videos_list),
        'next_page_token' => $response->getNextPageToken(),
        'prev_page_token' => $response->getPrevPageToken(),
        'channel_id' => $channel_info['channel_id'],
        'channel_title' => $channel_info['title'] ?? '',
      ];
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to list channel videos for @channel: @message', [
        '@channel' => $channel_input,
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }



  /**
   * Parses ISO 8601 duration to seconds.
   *
   * @param string $duration
   *   ISO 8601 duration string (e.g., PT1H2M30S).
   *
   * @return int
   *   Duration in seconds.
   */
  protected function parseIsoDuration(string $duration): int {
    $interval = new \DateInterval($duration);
    return ($interval->h * 3600) + ($interval->i * 60) + $interval->s;
  }

}
