<?php

declare(strict_types=1);

namespace Drupal\eca_google\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;

/**
 * Provides autocomplete functionality for Google API clients.
 */
class AutocompleteController extends ControllerBase {

  /**
   * Constructs an AutocompleteController object.
   */
  public function __construct() {
    // ControllerBase provides entityTypeManager() method
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static();
  }

  /**
   * Autocomplete API clients.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   The JSON response.
   */
  public function apiClients(Request $request): JsonResponse {
    $string = $request->query->get('q', '');
    $matches = [];

    if (!empty($string)) {
      $storage = $this->entityTypeManager()->getStorage('google_api_client');
      
      $query = $storage->getQuery()
        ->accessCheck(TRUE)
        ->condition('name', $string, 'CONTAINS')
        ->sort('name', 'ASC')
        ->range(0, 10);
      
      $entity_ids = $query->execute();
      
      if (!empty($entity_ids)) {
        $entities = $storage->loadMultiple($entity_ids);
        
        foreach ($entities as $entity) {
          $label_with_id = $entity->label() . ' (' . $entity->id() . ')';
          $matches[] = [
            'value' => $label_with_id,
            'label' => $label_with_id,
          ];
        }
      }
    }

    return new JsonResponse($matches);
  }

  /**
   * Autocomplete service accounts.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   The JSON response.
   */
  public function serviceAccounts(Request $request): JsonResponse {
    $string = $request->query->get('q', '');
    $matches = [];

    if (!empty($string)) {
      $storage = $this->entityTypeManager()->getStorage('google_api_service_client');
      
      // For config entities, use 'label' field instead of 'name'
      $query = $storage->getQuery()
        ->accessCheck(TRUE)
        ->condition('label', $string, 'CONTAINS')
        ->sort('label', 'ASC')
        ->range(0, 10);
      
      $entity_ids = $query->execute();
      
      if (!empty($entity_ids)) {
        $entities = $storage->loadMultiple($entity_ids);
        
        foreach ($entities as $entity) {
          $label_with_id = $entity->label() . ' (' . $entity->id() . ')';
          $matches[] = [
            'value' => $label_with_id,
            'label' => $label_with_id,
          ];
        }
      }
    }

    return new JsonResponse($matches);
  }

}