<?php

declare(strict_types=1);

namespace Drupal\eca_google;

/**
 * Trait for consistent datetime formatting across Google API services.
 */
trait DateTimeFormattingTrait {

  /**
   * Formats datetime value for Google API (RFC 3339).
   *
   * @param mixed $datetime
   *   The datetime value to format (string, DateTime, int, etc).
   * @param string|null $timezone
   *   Optional timezone identifier.
   * @param bool|null $all_day_format
   *   TRUE to force all-day format, FALSE to force datetime format, NULL for auto-detect.
   *
   * @return string|null
   *   RFC3339 formatted datetime string or NULL on failure.
   */
  public function formatDateTimeForApi(mixed $datetime, ?string $timezone = null, ?bool $all_day_format = NULL): ?string {
    if (empty($datetime)) {
      return NULL;
    }

    try {
      $dt = NULL;

      // Handle different input types
      if ($datetime instanceof \DateTime) {
        // Already a DateTime object (e.g., from Drupal field)
        $dt = clone $datetime;
      }
      elseif ($datetime instanceof \DateTimeInterface) {
        // Other DateTime interface implementations
        $dt = \DateTime::createFromInterface($datetime);
      }
      elseif (is_numeric($datetime)) {
        // Unix timestamp
        $dt = new \DateTime('@' . $datetime);
      }
      elseif (is_string($datetime)) {
        // String datetime - try various formats
        $dt = $this->parseStringDateTime($datetime);
      }
      elseif (is_array($datetime) && isset($datetime['date'])) {
        // Drupal field array format
        $dt = new \DateTime($datetime['date']);
      }

      if (!$dt) {
        if (method_exists($this, 'logger') && isset($this->logger)) {
          $this->logger->error('Unable to parse datetime value: @value', [
            '@value' => is_scalar($datetime) ? $datetime : gettype($datetime),
          ]);
        }
        return NULL;
      }

      // Determine formatting mode for Calendar API (all-day vs timed events)
      $use_all_day_format = $all_day_format;
      if ($use_all_day_format === NULL && method_exists($this, 'isAllDayEvent')) {
        // Auto-detect if not forced and method exists (Calendar service)
        $use_all_day_format = $this->isAllDayEvent($datetime, $timezone);
      }

      if ($use_all_day_format) {
        // For all-day events, return date-only format (YYYY-MM-DD)
        // Apply timezone first to get the correct date in the target timezone
        if (!empty($timezone)) {
          try {
            $dt->setTimezone(new \DateTimeZone($timezone));
          }
          catch (\Exception $e) {
            if (method_exists($this, 'logger') && isset($this->logger)) {
              $this->logger->warning('Invalid timezone "@tz", using datetime as-is: @message', [
                '@tz' => $timezone,
                '@message' => $e->getMessage(),
              ]);
            }
          }
        }
        return $dt->format('Y-m-d');
      }

      // For timed events, apply timezone if provided
      if (!empty($timezone)) {
        try {
          $dt->setTimezone(new \DateTimeZone($timezone));
        }
        catch (\Exception $e) {
          if (method_exists($this, 'logger') && isset($this->logger)) {
            $this->logger->warning('Invalid timezone "@tz", using datetime as-is: @message', [
              '@tz' => $timezone,
              '@message' => $e->getMessage(),
            ]);
          }
        }
      }

      // Return RFC3339 format for Google APIs
      // Note: 'c' and RFC3339 are equivalent formats
      return $dt->format('c');
    }
    catch (\Exception $e) {
      if (method_exists($this, 'logger') && isset($this->logger)) {
        $this->logger->error('Error formatting datetime for API: @message', [
          '@message' => $e->getMessage(),
        ]);
      }
      return NULL;
    }
  }

  /**
   * Parses string datetime with fallback formats.
   *
   * @param string $datetime
   *   The datetime string to parse.
   *
   * @return \DateTime|null
   *   Parsed DateTime object or NULL on failure.
   */
  private function parseStringDateTime(string $datetime): ?\DateTime {
    // Try general parsing first (handles most standard formats automatically)
    try {
      return new \DateTime($datetime);
    }
    catch (\Exception $e) {
      // General parsing failed, try specific formats for edge cases
    }

    // Specific formats for cases that general parsing can't handle
    $formats = [
      'D, d M Y - H:i',        // Fri, 12 Sep 2025 - 10:00 (Drupal default format)
      'd M Y - H:i',           // 12 Sep 2025 - 10:00 (without day name)
    ];

    // Try exact format matches for special cases
    foreach ($formats as $format) {
      $dt = \DateTime::createFromFormat($format, $datetime);
      if ($dt !== FALSE) {
        return $dt;
      }
    }

    return NULL;
  }

}
