<?php

declare(strict_types=1);

namespace Drupal\eca_google\Plugin\Action;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\eca\Plugin\Action\ConfigurableActionBase;
use Drupal\eca\Plugin\FormFieldPluginTrait;
use Drupal\eca_google\GoogleApiService;
use Symfony\Component\DependencyInjection\ContainerInterface;

abstract class GoogleApiActionBase extends ConfigurableActionBase {

  use FormFieldPluginTrait;

  /**
   * The Google API service.
   */
  protected GoogleApiService $googleApiService;

  /**
   * The entity type manager.
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->googleApiService = $container->get('eca_google.google_api');
    $instance->entityTypeManager = $container->get('entity_type.manager');
    return $instance;
  }

  protected function getAuthClientIdDefaultConfig() : array {
    return [
      'auth_type' => 'api_client',
      'auth_from' => 'direct',
      'api_client_id' => '',
      'service_account_id' => '',
      'auth_client_from_token' => '',
    ];
  }

  protected function addGoogleAuthConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form['auth_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Authentication Type'),
      '#description' => $this->t('Choose between API Client (OAuth2) or Service Account authentication.'),
      '#options' => [
        'api_client' => $this->t('API Client (OAuth2)'),
        'service_account' => $this->t('Service Account'),
      ],
      '#default_value' => $this->configuration['auth_type'],
      '#required' => TRUE,
    ];

    $form['auth_from'] = [
      '#type' => 'select',
      '#title' => $this->t('Authentication Source'),
      '#description' => $this->t('Choose whether to select authentication directly or use a token for dynamic selection.'),
      '#options' => [
        'direct' => $this->t('Select Directly'),
        'token' => $this->t('From Token'),
      ],
      '#default_value' => $this->configuration['auth_from'],
      '#required' => TRUE,
    ];

    $form['api_client_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('API Client'),
      '#description' => $this->t('Search and select an API Client for OAuth2 authentication.'),
      '#default_value' => $this->getApiClientDisplayValue($this->configuration['api_client_id']),
      '#autocomplete_route_name' => 'eca_google.autocomplete.api_clients',
      '#states' => [
        'visible' => [
          ':input[name="auth_type"]' => ['value' => 'api_client'],
          ':input[name="auth_from"]' => ['value' => 'direct'],
        ],
        'required' => [
          ':input[name="auth_type"]' => ['value' => 'api_client'],
          ':input[name="auth_from"]' => ['value' => 'direct'],
        ],
      ],
    ];

    $form['service_account_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Service Account'),
      '#description' => $this->t('Search and select a Service Account for server-to-server authentication.'),
      '#default_value' => $this->getServiceAccountDisplayValue($this->configuration['service_account_id']),
      '#autocomplete_route_name' => 'eca_google.autocomplete.service_accounts',
      '#states' => [
        'visible' => [
          ':input[name="auth_type"]' => ['value' => 'service_account'],
          ':input[name="auth_from"]' => ['value' => 'direct'],
        ],
        'required' => [
          ':input[name="auth_type"]' => ['value' => 'service_account'],
          ':input[name="auth_from"]' => ['value' => 'direct'],
        ],
      ],
    ];

    $form['auth_client_from_token'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Client ID from Token'),
      '#description' => $this->t('Token that resolves to a client ID (e.g., [user:field_google_client_id]). The token should contain just the client ID, not the auth type prefix.'),
      '#default_value' => $this->configuration['auth_client_from_token'],
      '#eca_token_replacement' => TRUE,
      '#states' => [
        'visible' => [
          ':input[name="auth_from"]' => ['value' => 'token'],
        ],
        'required' => [
          ':input[name="auth_from"]' => ['value' => 'token'],
        ],
      ],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state): void {
    parent::validateConfigurationForm($form, $form_state);

    $values = $form_state->getValues();
    $auth_type = $values['auth_type'] ?? '';
    $auth_from = $values['auth_from'] ?? '';

    // Validate that appropriate fields are filled based on selections
    if ($auth_from === 'direct') {
      if ($auth_type === 'api_client') {
        $api_client_value = $values['api_client_id'] ?? '';
        if (empty($api_client_value)) {
          $form_state->setErrorByName('api_client_id', $this->t('API Client selection is required.'));
        } elseif (!$this->extractClientIdFromAutocomplete($api_client_value)) {
          $form_state->setErrorByName('api_client_id', $this->t('Please select a valid API Client from the autocomplete suggestions.'));
        }
      } elseif ($auth_type === 'service_account') {
        $service_account_value = $values['service_account_id'] ?? '';
        if (empty($service_account_value)) {
          $form_state->setErrorByName('service_account_id', $this->t('Service Account selection is required.'));
        } elseif (!$this->extractClientIdFromAutocomplete($service_account_value)) {
          $form_state->setErrorByName('service_account_id', $this->t('Please select a valid Service Account from the autocomplete suggestions.'));
        }
      }
    } elseif ($auth_from === 'token') {
      $token_value = $values['auth_client_from_token'] ?? '';
      if (empty($token_value)) {
        $form_state->setErrorByName('auth_client_from_token', $this->t('Token field is required when using token-based authentication.'));
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    // Extract IDs from autocomplete values before storing in configuration
    $values = $form_state->getValues();

    if (!empty($values['api_client_id'])) {
      $extracted_id = $this->extractClientIdFromAutocomplete($values['api_client_id']);
      if ($extracted_id) {
        $this->configuration['api_client_id'] = $extracted_id;
      }
    }

    if (!empty($values['service_account_id'])) {
      $extracted_id = $this->extractClientIdFromAutocomplete($values['service_account_id']);
      if ($extracted_id) {
        $this->configuration['service_account_id'] = $extracted_id;
      }
    }
    parent::submitConfigurationForm($form, $form_state);
  }

  /**
   * Gets display value for API client field.
   *
   * @param string $client_id
   *   The stored client ID.
   *
   * @return string
   *   The display value in "Label (ID)" format, or just the ID if entity not found.
   */
  protected function getApiClientDisplayValue(string $client_id): string {
    if (empty($client_id)) {
      return '';
    }
    
    try {
      $storage = $this->entityTypeManager->getStorage('google_api_client');
      $entity = $storage->load($client_id);
      
      if ($entity) {
        return $entity->label() . ' (' . $entity->id() . ')';
      }
    } catch (\Exception $e) {
      // If there's an error loading the entity, fall back to just the ID
    }
    
    return $client_id;
  }

  /**
   * Gets display value for service account field.
   *
   * @param string $client_id
   *   The stored client ID.
   *
   * @return string
   *   The display value in "Label (ID)" format, or just the ID if entity not found.
   */
  protected function getServiceAccountDisplayValue(string $client_id): string {
    if (empty($client_id)) {
      return '';
    }
    
    try {
      $storage = $this->entityTypeManager->getStorage('google_api_service_client');
      $entity = $storage->load($client_id);
      
      if ($entity) {
        return $entity->label() . ' (' . $entity->id() . ')';
      }
    } catch (\Exception $e) {
      // If there's an error loading the entity, fall back to just the ID
    }
    
    return $client_id;
  }

  /**
   * Extracts client ID from autocomplete format "Label (client_id)".
   *
   * @param string $autocomplete_value
   *   The value from autocomplete field.
   *
   * @return string|null
   *   The extracted client ID or NULL if not found.
   */
  protected function extractClientIdFromAutocomplete(string $autocomplete_value): ?string {
    // Match pattern "anything (client_id)"
    if (preg_match('/\(([^)]+)\)$/', $autocomplete_value, $matches)) {
      return trim($matches[1]);
    }

    // If no parentheses found, assume it's already just the ID
    return trim($autocomplete_value) ?: NULL;
  }

  /**
   * Validates and resolves authentication configuration to auth_type and client_id.
   *
   * @param string $operation
   *   The operation name for logging context.
   *
   * @return array|null
   *   Array containing ['auth_type', 'client_id'] or NULL if invalid/missing.
   */
  protected function validateAndParseAuth(string $operation): ?array {
    $auth_type = $this->configuration['auth_type'];
    $auth_from = $this->configuration['auth_from'];

    if (empty($auth_type) || empty($auth_from)) {
      $this->logger->error('Google @operation action: Missing authentication configuration.', [
        '@operation' => $operation,
      ]);
      return NULL;
    }

    // Resolve client ID based on auth_from setting
    if ($auth_from === 'direct') {
      // Use direct selection - configuration already contains just the ID
      if ($auth_type === 'api_client') {
        $client_id = $this->configuration['api_client_id'];
      } elseif ($auth_type === 'service_account') {
        $client_id = $this->configuration['service_account_id'];
      } else {
        $this->logger->error('Google @operation action: Invalid auth_type "@type".', [
          '@operation' => $operation,
          '@type' => $auth_type,
        ]);
        return NULL;
      }
    } elseif ($auth_from === 'token') {
      // Resolve from token - tokens contain ID only
      $token_value = $this->configuration['auth_client_from_token'];
      if (empty($token_value)) {
        $this->logger->error('Google @operation action: Token auth selected but no token provided.', [
          '@operation' => $operation,
        ]);
        return NULL;
      }
      $client_id = $this->tokenService->replacePlain($token_value, [], ['clear' => TRUE]);
    } else {
      $this->logger->error('Google @operation action: Invalid auth_from "@from".', [
        '@operation' => $operation,
        '@from' => $auth_from,
      ]);
      return NULL;
    }

    // Validate client ID is not empty
    if (empty($client_id)) {
      $this->logger->error('Google @operation action: Resolved client ID is empty for @operation.', [
        '@operation' => $operation,
      ]);
      return NULL;
    }

    return [$auth_type, $client_id];
  }
}
