<?php

namespace Drupal\eca_group\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\TypedData\TypedDataInterface;
use Drupal\eca\Attribute\EcaAction;
use Drupal\eca_base\Plugin\Action\ListCompare;
use Drupal\group\Entity\GroupMembershipInterface;

/**
 * Action to compare group members in two lists of memberships.
 */
#[Action(
  id: 'eca_group_list_compare_memberships',
  label: new TranslatableMarkup('List: compare members in memberships'),
)]
#[EcaAction(
  description: new TranslatableMarkup('Compares the members in two lists of memberships.'),
  version_introduced: '1.0.0',
)]
class ListCompareMemberships extends ListCompare {

  /**
   * Cleans the list to only contain group memberships.
   *
   * @param iterable $list
   *   The list of data assuming to contain group memberships.
   *
   * @return \Drupal\group\Entity\GroupMembershipInterface[]
   *   The cleaned list of group memberships.
   */
  protected function cleanupList(iterable $list): array {
    $newList = [];
    foreach ($list as $value) {
      if ($value instanceof GroupMembershipInterface) {
        $newList[] = $value;
      }
      elseif ($value instanceof TypedDataInterface && $value->getValue() instanceof GroupMembershipInterface) {
        $newList[] = $value->getValue();
      }
    }
    return $newList;
  }

  /**
   * Receives a token and counts the contained items.
   *
   * @param iterable $list1
   *   First list to compare.
   * @param iterable $list2
   *   Secondary list to compare.
   *
   * @return array
   *   Result of the array_diff
   */
  protected function getDiff(iterable $list1, iterable $list2): array {
    $result = [];
    $list1 = $this->cleanupList($list1);
    $list2 = $this->cleanupList($list2);
    foreach ($list1 as $value1) {
      $found = FALSE;
      foreach ($list2 as $value2) {
        if ($value1->getEntity()->id() === $value2->getEntity()->id()) {
          $found = TRUE;
          break;
        }
      }
      if (!$found) {
        $result[] = $value1;
      }
    }
    return $result;
  }

  /**
   * Receives a token and counts the contained items.
   *
   * @param iterable $list1
   *   First list to compare.
   * @param iterable $list2
   *   Secondary list to compare.
   *
   * @return array
   *   Result of the array_intersect
   */
  protected function getIntersect(iterable $list1, iterable $list2): array {
    $result = [];
    $list1 = $this->cleanupList($list1);
    $list2 = $this->cleanupList($list2);
    foreach ($list1 as $value1) {
      foreach ($list2 as $value2) {
        if ($value1->getEntity()->id() === $value2->getEntity()->id()) {
          $result[] = $value1;
          break;
        }
      }
    }
    return $result;
  }

}
