<?php

namespace Drupal\eca_group\Plugin\ECA\Condition;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Extension\ModuleExtensionList;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\Context\EntityContextDefinition;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaCondition;
use Drupal\eca\Plugin\ECA\Condition\ConditionBase;
use Drupal\group\Access\GroupPermissionHandlerInterface;
use Drupal\group\Entity\Group;
use Drupal\group\Entity\GroupInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin of the ECA condition if current user has group permission.
 */
#[EcaCondition(
  id: 'eca_group_current_user_has_group_permission',
  label: new TranslatableMarkup('Group: current user has permission'),
  context_definitions: [
    'entity' => new EntityContextDefinition(
      data_type: 'entity:group',
      label: new TranslatableMarkup('Group'),
    ),

  ],
  description: new TranslatableMarkup('Verifies if the current user has the given permission in a given group.'),
  version_introduced: '1.0.0',
)]
class CurrentUserHasGroupPermission extends ConditionBase {

  /**
   * The group permission handler.
   *
   * @var \Drupal\group\Access\GroupPermissionHandlerInterface
   */
  protected GroupPermissionHandlerInterface $permissionHandler;

  /**
   * The module extension list.
   *
   * @var \Drupal\Core\Extension\ModuleExtensionList
   */
  protected ModuleExtensionList $moduleHandler;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->permissionHandler = $container->get('group.permissions');
    $instance->moduleHandler = $container->get('extension.list.module');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function evaluate(): bool {
    $result = AccessResult::forbidden();
    $group = $this->getValueFromContext('entity');
    if (is_scalar($group)) {
      $group = Group::load($group);
    }
    if ($group instanceof GroupInterface) {
      $result = $group->hasPermission($this->configuration['permission'], $this->currentUser);
    }
    return $this->negationCheck($result);
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return ['permission' => ''] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $permissions = [];
    foreach ($this->permissionHandler->getPermissions(TRUE) as $permission_name => $permission) {
      $display_name = $this->moduleHandler->getName($permission['provider']);
      $permissions[$display_name . ' : ' . $permission['section']][$permission_name] = strip_tags($permission['title']);
    }
    $form['permission'] = [
      '#type' => 'select',
      '#title' => $this->t('Permission'),
      '#description' => $this->t('The group permission to verify.'),
      '#default_value' => $this->configuration['permission'],
      '#options' => $permissions,
      '#required' => TRUE,
    ];
    return parent::buildConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    $this->configuration['permission'] = $form_state->getValue('permission');
    parent::submitConfigurationForm($form, $form_state);
  }

}
