<?php

namespace Drupal\eca_helper\Plugin\Action;

use Drupal\Core\Form\FormStateInterface;
use Drupal\eca\Plugin\Action\ConfigurableActionBase;

/**
 * Date Format To Unix.
 *
 * @Action(
 *   id = "eca_helper_date_format_to_unix",
 *   label = @Translation("ECA Helper: Date Format To Unix"),
 *   description = @Translation("Action convert date time format to unix
 *   time."),
 * )
 */
class DateFormatToUnix extends ConfigurableActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'date' => '',
      'format' => '',
      'token_name' => '',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);
    $form['date'] = [
      '#type' => 'textfield',
      '#required' => TRUE,
      '#title' => $this->t('Date time value'),
      '#default_value' => $this->configuration['date'],
    ];

    $form['format'] = [
      '#type' => 'textfield',
      '#required' => TRUE,
      '#title' => $this->t('Date time format'),
      '#default_value' => $this->configuration['format'],
      '#description' => $this->t('A user-defined php date format string like "m-d-Y H:i". See the PHP manual for available options. Use "|" end of format like| "Y-F-d|" for compatible with strtotime.'),
    ];

    $form['token_name'] = [
      '#required' => TRUE,
      '#type' => 'textfield',
      '#title' => $this->t('Name of response token'),
      '#default_value' => $this->configuration['token_name'],
      '#description' => $this->t('The response value will be loaded into this specified token.'),
      '#eca_token_reference' => TRUE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    $this->configuration['date'] = $form_state->getValue('date');
    $this->configuration['format'] = $form_state->getValue('format');
    $this->configuration['token_name'] = $form_state->getValue('token_name');
    parent::submitConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function execute() {
    if (!empty($this->configuration['token_name'])) {
      $date = $this->tokenService->replaceClear($this->configuration['date']);
      $format = $this->tokenService->replaceClear($this->configuration['format']);
      $date = \DateTime::createFromFormat($format, $date);

      if ($date) {
        $this->tokenService->addTokenData($this->configuration['token_name'], $date->getTimestamp());
      }
    }
  }

}
