<?php

namespace Drupal\eca_helper\Event;

use Drupal\Component\EventDispatcher\Event;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\eca\Event\AccessEventInterface;
use Drupal\file\FileInterface;

/**
 * Defines the file download event.
 */
class FileDownloadEvent extends Event implements AccessEventInterface {

  /**
   * File download event.
   */
  const FILE_DOWNLOAD = 'eca_helper.file_download';

  /**
   * The file URI.
   */
  protected string $uri;

  /**
   * The file entity.
   */
  protected ?FileInterface $file;

  /**
   * The account that asks for access.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected AccountInterface $account;

  /**
   * The access result.
   */
  protected ?AccessResultInterface $accessResult;

  /**
   * Constructs a new FileDownloadEvent object.
   *
   * @param string $uri
   *   The file URI.
   * @param \Drupal\file\FileInterface|null $file
   *   The file entity.
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The account that asks for access.
   */
  public function __construct(string $uri, ?FileInterface $file, AccountInterface $account) {
    $this->uri = $uri;
    $this->file = $file;
    $this->account = $account;
    $this->accessResult = AccessResult::neutral();
  }

  /**
   * Gets the file URI.
   *
   * @return string
   *   The file URI.
   */
  public function getUri(): string {
    return $this->uri;
  }

  /**
   * Gets the file entity.
   *
   * @return \Drupal\file\FileInterface|null
   *   The file entity.
   */
  public function getFile(): ?FileInterface {
    return $this->file;
  }

  /**
   * {@inheritdoc}
   */
  public function setAccessResult(AccessResultInterface $result): AccessEventInterface {
    $this->accessResult = $result;
    return $this;
  }

  /**
   * Gets the access result.
   *
   * @return \Drupal\Core\Access\AccessResultInterface|null
   *   The access result.
   */
  public function getAccessResult(): ?AccessResultInterface {
    return $this->accessResult;
  }

  /**
   * {@inheritdoc}
   */
  public function getAccount(): AccountInterface {
    return $this->account;
  }

}
