<?php

namespace Drupal\eca_helper\Service;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\eca\Event\AccessEventInterface;
use Drupal\eca\Event\TriggerEvent;
use Drupal\Core\Access\AccessResultInterface;

/**
 * Manages file download access logic by dispatching an ECA event.
 */
class FileDownloadManager {

  /**
   * The entity type manager.
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The ECA trigger event.
   */
  protected TriggerEvent $triggerEvent;

  /**
   * The current user.
   */
  protected AccountInterface $currentUser;

  /**
   * Constructs a new FileDownloadManager object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\eca\Event\TriggerEvent $trigger_event
   *   The ECA trigger event.
   * @param \Drupal\Core\Session\AccountInterface $current_user
   *   The current user.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager, TriggerEvent $trigger_event, AccountInterface $current_user) {
    $this->entityTypeManager = $entity_type_manager;
    $this->triggerEvent = $trigger_event;
    $this->currentUser = $current_user;
  }

  /**
   * Handles the logic for hook_file_download().
   *
   * This method is called by the hook implementation in the .module file.
   * It loads the file entity, dispatches the ECA event, and returns the
   * appropriate access result.
   *
   * @param string $uri
   *   The URI of the file being downloaded.
   *
   * @return int|array|null
   *   A value suitable for hook_file_download()'s return value. Returns -1
   *   to deny access, an array of headers to grant access, or NULL to abstain.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function onFileDownload(string $uri): array|int|null {
    // We only care about private files.
    if (!str_starts_with($uri, 'private://')) {
      return NULL;
    }

    // Load the file entity associated with the URI.
    $files = $this->entityTypeManager
      ->getStorage('file')
      ->loadByProperties(['uri' => $uri]);
    $file = !empty($files) ? reset($files) : NULL;

    // Dispatch the event to be handled by ECA.
    $event = $this->triggerEvent->dispatchFromPlugin('eca_helper_file_download:private_file', $uri, $file, $this->currentUser);
    if (!($event instanceof AccessEventInterface)) {
      return NULL;
    }

    // Get the access result that was set on the event by an ECA model.
    $access_result = $event->getAccessResult();

    if ($access_result instanceof AccessResultInterface) {
      if ($access_result->isForbidden()) {
        return -1;
      }
      if ($access_result->isAllowed()) {
        // Returning an empty array of headers allows the download. Other
        // modules can still add their own headers.
        return [];
      }
    }

    // If no ECA model made a decision, we abstain.
    return NULL;
  }

}
