<?php

namespace Drupal\eca_helper\Plugin\Action;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\TypedData\Plugin\DataType\StringData;
use Drupal\eca\Plugin\Action\ConfigurableActionBase;
use Drupal\eca_helper\Traits\DumperTrait;
use Drupal\user\Entity\Role;

/**
 * Set access to a form field.
 *
 * @Action(
 *   id = "eca_helper_dumper",
 *   label = @Translation("ECA Helper: Dumper"),
 *   description = @Translation("Dump the data.")
 * )
 */
class Dumper extends ConfigurableActionBase {

  use DumperTrait;
  use Helper;

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'data' => '',
      'data2' => '',
      'data3' => '',
      'data4' => '',
      'role' => 'administrator',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);
    for ($i = 1; $i <= 4; $i++) {
      $data_id = $i === 1 ? 'data' : ('data' . $i);
      $form[$data_id] = [
        '#type' => 'textarea',
        '#title' => $this->t('Value @num', ['@num' => $i]),
        '#required' => FALSE,
        '#description' => $this->t('The data used for dumping.'),
        '#default_value' => $this->configuration[$data_id],
        '#eca_token_replacement' => TRUE,
      ];
    }

    // Role element.
    $roles = Role::loadMultiple();
    $role_options = [];

    foreach ($roles as $role_id => $role) {
      $role_options[$role_id] = $role->label();
    }
    $form['role'] = [
      '#type' => 'select',
      '#required' => TRUE,
      '#title' => $this->t('Role'),
      '#description' => $this->t('The role used for dumping.'),
      '#default_value' => $this->configuration['role'],
      '#options' => $role_options,
    ];
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    for ($i = 1; $i <= 4; $i++) {
      $data_id = $i === 1 ? 'data' : ('data' . $i);
      $this->configuration[$data_id] = $form_state->getValue($data_id);
    }
    $this->configuration['role'] = $form_state->getValue('role');
    parent::submitConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function execute() {
    $role = $this->tokenService->getOrReplace($this->configuration['role']) ?? 'administrator';
    if (!$this->currentUser->hasRole($role)) {
      return;
    }

    for ($i = 1; $i <= 4; $i++) {
      $data_id = $i === 1 ? 'data' : ('data' . $i);
      if (!(is_string($this->configuration[$data_id]) && strlen($this->configuration[$data_id]) > 0)) {
        continue;
      }
      $value = $this->getTokenValue($this->configuration[$data_id]);
      if (is_scalar($value) || (is_object($value) && method_exists($value, '__toString'))) {
        $value = (string) $value;
      }

      if ($value instanceof StringData) {
        $value = $value->getString();
      }

      $this->dump($value);
    }
  }

}
