<?php

namespace Drupal\eca_helper\Plugin\Action;

use Drupal\Core\Form\FormStateInterface;
use Drupal\eca\Plugin\Action\ConfigurableActionBase;

/**
 * ECA action for Json encode/decode.
 *
 * @Action(
 *   id = "eca_helper_json_encode_decode",
 *   label = @Translation("ECA Helper: Json Encode/Decode"),
 *   description = @Translation("Encode and decode json."),
 * )
 */
class Json extends ConfigurableActionBase {

  use Helper;

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'data' => '',
      'type' => 'encode',
      'token_name' => '',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);
    $form['data'] = [
      '#type' => 'textarea',
      '#required' => TRUE,
      '#title' => $this->t('Data'),
      '#description' => $this->t('The date used for encode/decode. This support token replacement.'),
      '#cols' => 60,
      '#rows' => 5,
      '#default_value' => $this->configuration['token_name'],
    ];
    $form['type'] = [
      '#type' => 'select',
      '#required' => TRUE,
      '#title' => $this->t('Action type'),
      '#default_value' => $this->configuration['type'],
      '#options' => [
        'encode' => $this->t('Encode'),
        'decode' => $this->t('Decode'),
      ],
    ];
    $form['token_name'] = [
      '#required' => TRUE,
      '#type' => 'textfield',
      '#title' => $this->t('Name of token'),
      '#default_value' => $this->configuration['token_name'],
      '#description' => $this->t('The encode/decode value will be loaded into this specified token.'),
      '#eca_token_reference' => TRUE,
    ];
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    $this->configuration['type'] = $form_state->getValue('type');
    $this->configuration['data'] = $form_state->getValue('data');
    $this->configuration['token_name'] = $form_state->getValue('token_name');
    parent::submitConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function execute() {
    $value = $this->getTokenValue($this->configuration['data']);
    $type = $this->configuration['type'];

    try {
      if ($type === 'encode') {
        $result = json_encode($value, TRUE);
      }
      else {
        !is_string($value) && ($value = (string) $value);
        $result = json_decode($value);
      }
      if (!empty($this->configuration['token_name'])) {
        $this->tokenService->addTokenData($this->configuration['token_name'], $result);
      }
    }
    catch (\Exception $e) {
      $this->logger
        ->error('ECA Helper: Json Encode/Decode: ' . $e->getMessage());
    }
  }

}
