<?php

namespace Drupal\eca_helper\Plugin\Action;

use Drupal\Core\Form\FormStateInterface;
use Drupal\eca\Plugin\Action\ConfigurableActionBase;

/**
 * Provides an 'Eca Helper: File read' action.
 *
 * This action allows reading file content from a specified path (public:// or
 * private://) and saves the content to a token for use in subsequent ECA
 * actions.
 *
 * @Action(
 * id = "eca_helper_file_read",
 * label = @Translation("Eca Helper: File raw read"),
 * description = @Translation("Reads content from a file (public:// or
 *   private://) and saves it to a token.")
 * )
 */
class FileReadAction extends ConfigurableActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'file_path' => 'public://',
      'token_name' => 'file_content',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['file_path'] = [
      '#type' => 'textfield',
      '#title' => $this->t('File path'),
      '#description' => $this->t('Enter the path to the file (e.g., public://myfile.txt or private://secret.doc). Tokens are allowed.'),
      '#default_value' => $this->configuration['file_path'],
      '#required' => TRUE,
    ];
    $form['token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Token name for content'),
      '#description' => $this->t('Enter the name for the token where the file content will be stored (e.g., "my_file_data").'),
      '#default_value' => $this->configuration['token_name'],
      '#required' => TRUE,
      '#eca_token_reference' => TRUE,
    ];
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    $this->configuration['file_path'] = $form_state->getValue('file_path');
    $this->configuration['token_name'] = $form_state->getValue('token_name');
    parent::submitConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $file_path_raw = $this->configuration['file_path'] ?? '';
    $token_name = $this->configuration['token_name'] ?? 'file_content';

    $file_path = $this->tokenService->getOrReplace($file_path_raw);

    if (empty($file_path)) {
      $this->logger->warning('File read action: File path is empty.');
      $this->tokenService->addTokenData($token_name, '');
      return;
    }

    if (!str_starts_with($file_path, 'public://') && !str_starts_with($file_path, 'private://')) {
      $this->logger->warning('File read action: File path "{path}" must start with public:// or private://.', ['path' => $file_path]);
      $this->tokenService->addTokenData($token_name, '');
      return;
    }

    if (!file_exists($file_path)) {
      $this->logger->warning('File read action: File "{path}" does not exist.', ['path' => $file_path]);
      $this->tokenService->addTokenData($token_name, '');
      return;
    }

    $file_content = @file_get_contents($file_path);

    if ($file_content === FALSE) {
      $this->logger->error('File read action: Could not read file content from "{path}".', ['path' => $file_path]);
      $this->tokenService->addTokenData($token_name, '');
    }
    else {
      $this->tokenService->addTokenData($token_name, $file_content);
      $this->logger->info('File read action: Successfully read file "{path}" and stored content in token data stack as "{token}".', [
        'path' => $file_path,
        'token' => $token_name,
      ]);
    }
  }

}
