<?php

namespace Drupal\eca_helper\Decorate;

use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\eca\Event\TriggerEvent;

/**
 * Decorates the core messenger service to add custom functionality.
 */
class Messenger implements MessengerInterface {

  use StringTranslationTrait;

  /**
   * The original messenger service.
   */
  protected MessengerInterface $originalMessenger;

  /**
   * The ECA trigger event.
   */
  protected TriggerEvent $triggerEvent;

  /**
   * Constructs a CustomMessenger object.
   *
   * @param \Drupal\Core\Messenger\MessengerInterface $original_messenger
   *   The original messenger service.
   * @param \Drupal\eca\Event\TriggerEvent $trigger_event
   *   The ECA trigger event.
   */
  public function __construct(MessengerInterface $original_messenger, TriggerEvent $trigger_event) {
    $this->originalMessenger = $original_messenger;
    $this->triggerEvent = $trigger_event;
  }

  /**
   * {@inheritdoc}
   */
  public function addMessage($message, $type = self::TYPE_STATUS, $repeat = FALSE) {
    if ($message instanceof \Stringable) {
      try {
        /** @var \Drupal\eca_helper\Event\StatusMessagesEvent $event */
        $msg = (string) $message;
        $event = $this->triggerEvent->dispatchFromPlugin('eca_helper:status_messages', $msg, $type);
        if ($event->isChanged()) {
          $message_altered = $event->getMessage();
          if ($message_altered && mb_strlen($message) > 0) {
            return $this->originalMessenger->addMessage($message_altered, $type, $repeat);
          }
          return $this;
        }
      }
      catch (\Exception $e) {
        // During early bootstrap (e.g., installation), some services may not
        // be available yet. In this case, fall back to the original messenger.
        return $this->originalMessenger->addMessage($message, $type, $repeat);
      }
    }
    return $this->originalMessenger->addMessage($message, $type, $repeat);
  }

  /**
   * {@inheritdoc}
   */
  public function addStatus($message, $repeat = FALSE) {
    return $this->addMessage($message, static::TYPE_STATUS, $repeat);
  }

  /**
   * {@inheritdoc}
   */
  public function addError($message, $repeat = FALSE) {
    return $this->addMessage($message, static::TYPE_ERROR, $repeat);
  }

  /**
   * {@inheritdoc}
   */
  public function addWarning($message, $repeat = FALSE) {
    return $this->addMessage($message, static::TYPE_WARNING, $repeat);
  }

  /**
   * {@inheritdoc}
   */
  public function all() {
    return $this->originalMessenger->all();
  }

  /**
   * {@inheritdoc}
   */
  public function deleteAll() {
    return $this->originalMessenger->deleteAll();
  }

  /**
   * {@inheritdoc}
   */
  public function messagesByType($type) {
    if (method_exists($this->originalMessenger, 'messagesByType')) {
      return $this->originalMessenger->messagesByType($type);
    }
    $messages = $this->all();
    return $messages[$type] ?? [];
  }

  /**
   * {@inheritdoc}
   */
  public function deleteByType($type) {
    return $this->originalMessenger->deleteByType($type);
  }

}
