<?php

namespace Drupal\eca_helper\Plugin\Action;

use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\eca\Attribute\EcaAction;
use Drupal\eca\Plugin\Action\ConfigurableActionBase;

/**
 * Provides an 'Eca Helper: File raw exist' action.
 *
 * This action check file exist from a specified path (public:// or
 * private://) and saves the result to a token for use in subsequent ECA
 * actions.
 */
#[Action(
  id: 'eca_helper_file_exist',
  label: new TranslatableMarkup('Eca Helper: File raw exist')
)]
#[EcaAction(
  description: new TranslatableMarkup('Check file exist from a file (public:// or
   private://) and saves it to a token.')
)]
class FileExistAction extends ConfigurableActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'file_path' => 'public://',
      'token_name' => 'file_content',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['file_path'] = [
      '#type' => 'textfield',
      '#title' => $this->t('File path'),
      '#description' => $this->t('Enter the path to the file (e.g., public://myfile.txt or private://secret.doc). Tokens are allowed.'),
      '#default_value' => $this->configuration['file_path'],
      '#required' => TRUE,
    ];
    $form['token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Token name for content'),
      '#description' => $this->t('Enter the name for the token where the file content will be stored (e.g., "file_exist"). Return 1 if file exist otherwise 0.'),
      '#default_value' => $this->configuration['token_name'],
      '#required' => TRUE,
      '#eca_token_reference' => TRUE,
    ];
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    $this->configuration['file_path'] = $form_state->getValue('file_path');
    $this->configuration['token_name'] = $form_state->getValue('token_name');
    parent::submitConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $file_path_raw = $this->configuration['file_path'] ?? '';
    $token_name = $this->configuration['token_name'] ?? 'file_content';

    $file_path = $this->tokenService->getOrReplace($file_path_raw);

    if (empty($file_path)) {
      $this->logger->warning('File raw exist action: File path is empty.');
      $this->tokenService->addTokenData($token_name, 0);
      return;
    }

    if (!str_starts_with($file_path, 'public://') && !str_starts_with($file_path, 'private://')) {
      $this->logger->warning('File raw exist action: File path "{path}" must start with public:// or private://.', ['path' => $file_path]);
      $this->tokenService->addTokenData($token_name, '');
      return;
    }

    $this->tokenService->addTokenData($token_name, file_exists($file_path) ? 1 : 0);
  }

}
