<?php

namespace Drupal\eca_helper\Plugin\Action;

use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\File\FileExists;
use Drupal\Core\File\FileSystem;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\eca\Attribute\EcaAction;
use Drupal\eca\Plugin\Action\ConfigurableActionBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides an 'Eca Helper: File raw write' action.
 *
 * This action allows writing data to a file (public:// or private://) and
 * makes the actual path of the written file available as a token.
 */
#[Action(
  id: 'eca_helper_file_write',
  label: new TranslatableMarkup('Eca Helper: File raw write')
)]
#[EcaAction(
  description: new TranslatableMarkup('Writes data to a file and provides
  the resulting file path as a token.')
)]
class FileWriteAction extends ConfigurableActionBase {

  use StringTranslationTrait;

  /**
   * The file system instance.
   */
  protected FileSystem $fileSystem;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->fileSystem = $container->get('file_system');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'file_path' => 'public://output.txt',
      'file_content' => '',
      'write_mode' => 'replace',
      'result_file_path_token_name' => 'written_file_path',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['file_path'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Target file path'),
      '#description' => $this->t('Enter the path where the file will be written (e.g., public://output.txt). Tokens are allowed.'),
      '#default_value' => $this->configuration['file_path'],
      '#required' => TRUE,
    ];
    $form['file_content'] = [
      '#type' => 'textarea',
      '#title' => $this->t('File content'),
      '#description' => $this->t('Enter the content to write to the file. Tokens are allowed.'),
      '#default_value' => $this->configuration['file_content'],
    ];
    $form['write_mode'] = [
      '#type' => 'select',
      '#title' => $this->t('Write mode'),
      '#description' => $this->t('What to do if the file already exists.'),
      '#options' => [
        'replace' => $this->t('Replace existing file'),
        'rename' => $this->t('Rename to a new unique name (e.g., output_0.txt)'),
      ],
      '#default_value' => $this->configuration['write_mode'],
    ];
    $form['result_file_path_token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Token name for resulting file path'),
      '#description' => $this->t('Enter the name for the token where the actual path of the written file will be stored (e.g., "actual_output_path").'),
      '#default_value' => $this->configuration['result_file_path_token_name'],
      '#required' => TRUE,
      '#eca_token_reference' => TRUE,
    ];
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    $this->configuration['file_path'] = $form_state->getValue('file_path');
    $this->configuration['file_content'] = $form_state->getValue('file_content');
    $this->configuration['write_mode'] = $form_state->getValue('write_mode');
    $this->configuration['result_file_path_token_name'] = $form_state->getValue('result_file_path_token_name');
    parent::submitConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $file_path_raw = $this->configuration['file_path'] ?? '';
    $file_content_raw = $this->configuration['file_content'] ?? '';
    $write_mode = $this->configuration['write_mode'] ?? 'replace';
    $result_token_name = $this->configuration['result_file_path_token_name'] ?? 'written_file_path';

    // Replace tokens using the token service from the base class.
    $file_path = $this->tokenService->getOrReplace($file_path_raw);
    $file_content = $this->tokenService->getOrReplace($file_content_raw);

    if (empty($file_path)) {
      $this->logger->warning('File write action: File path is empty.');
      $this->tokenService->addTokenData($result_token_name, '');
      return;
    }

    if (!str_starts_with($file_path, 'public://') && !str_starts_with($file_path, 'private://')) {
      $this->logger->warning('File write action: File path "{path}" must start with public:// or private://.', ['path' => $file_path]);
      $this->tokenService->addTokenData($result_token_name, '');
      return;
    }

    $directory = $this->fileSystem->dirname($file_path);
    if (!$this->fileSystem->prepareDirectory($directory, FileSystemInterface::CREATE_DIRECTORY | FileSystemInterface::MODIFY_PERMISSIONS)) {
      $this->logger->error('File write action: Could not prepare directory "{dir}" for writing.', ['dir' => $directory]);
      $this->tokenService->addTokenData($result_token_name, '');
      return;
    }

    $mode = FileExists::Replace;
    if ($write_mode === 'rename') {
      $mode = FileExists::Rename;
    }
    $final_file_path = $this->fileSystem->saveData($file_content, $file_path, $mode);

    if ($final_file_path) {
      $this->logger->info('File write action: Successfully wrote data to "{path}".', ['path' => $final_file_path]);
      $this->tokenService->addTokenData($result_token_name, $final_file_path);
    }
    else {
      $this->logger->error('File write action: Could not write data to "{path}". Attempted original path was {original_path}', [
        'path' => $file_path,
        'original_path' => $file_path_raw,
      ]);
      $this->tokenService->addTokenData($result_token_name, '');
    }
  }

}
