<?php

declare(strict_types=1);

namespace Drupal\eca_hubspot\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;

/**
 * Associate two HubSpot objects.
 */
#[Action(
  id: 'eca_hubspot_associate_objects',
  label: new TranslatableMarkup('HubSpot: Associate Objects'),
  category: new TranslatableMarkup('HubSpot'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Create an association between two objects in HubSpot.'),
  version_introduced: '1.0.0',
)]
final class AssociateObjects extends HubSpotActionBase {

  public function defaultConfiguration(): array {
    return [
      'from_object_type' => '',
      'from_object_id' => '',
      'to_object_type' => '',
      'to_object_id' => '',
    ] + parent::defaultConfiguration();
  }

  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['from_object_type'] = [
      '#type' => 'textfield',
      '#title' => $this->t('From Object Type'),
      '#description' => $this->t('The source object type (e.g., "contact", "company", "deal", "ticket", "lead").'),
      '#default_value' => $this->configuration['from_object_type'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['from_object_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('From Object ID'),
      '#description' => $this->t('The source object ID.'),
      '#default_value' => $this->configuration['from_object_id'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['to_object_type'] = [
      '#type' => 'textfield',
      '#title' => $this->t('To Object Type'),
      '#description' => $this->t('The target object type (e.g., "contact", "company", "deal", "ticket", "lead").'),
      '#default_value' => $this->configuration['to_object_type'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['to_object_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('To Object ID'),
      '#description' => $this->t('The target object ID.'),
      '#default_value' => $this->configuration['to_object_id'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    return $form;
  }

  public function execute(): void {
    $from_object_type = $this->tokenService->replacePlain($this->configuration['from_object_type']);
    $from_object_id = $this->tokenService->replacePlain($this->configuration['from_object_id']);
    $to_object_type = $this->tokenService->replacePlain($this->configuration['to_object_type']);
    $to_object_id = $this->tokenService->replacePlain($this->configuration['to_object_id']);

    if (empty($from_object_type) || empty($from_object_id) || empty($to_object_type) || empty($to_object_id)) {
      $this->logError('All fields are required to associate objects.');
      return;
    }

    $success = $this->hubspotService->associateObjects(
      $from_object_type,
      $from_object_id,
      $to_object_type,
      $to_object_id
    );

    if (!$success) {
      $this->logError('Failed to associate @from_type:@from_id with @to_type:@to_id', [
        '@from_type' => $from_object_type,
        '@from_id' => $from_object_id,
        '@to_type' => $to_object_type,
        '@to_id' => $to_object_id,
      ]);
    }
  }

}
