<?php

declare(strict_types=1);

namespace Drupal\eca_hubspot\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;

/**
 * Create a HubSpot deal.
 */
#[Action(
  id: 'eca_hubspot_create_deal',
  label: new TranslatableMarkup('HubSpot: Create Deal'),
  category: new TranslatableMarkup('HubSpot'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Create a new deal (opportunity) in HubSpot pipeline.'),
  version_introduced: '1.0.0',
)]
final class CreateDeal extends HubSpotActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'dealname' => '',
      'dealstage' => '',
      'pipeline' => '',
      'amount' => '',
      'closedate' => '',
      'associated_contact_ids' => '',
      'associated_company_ids' => '',
      'additional_properties' => '',
      'token_name' => '',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['dealname'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Deal Name'),
      '#description' => $this->t('The name of the deal.'),
      '#default_value' => $this->configuration['dealname'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['dealstage'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Deal Stage'),
      '#description' => $this->t('The deal stage ID (e.g., "appointmentscheduled", "qualifiedtobuy", "closedwon"). You can find stage IDs in your HubSpot pipeline settings.'),
      '#default_value' => $this->configuration['dealstage'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['pipeline'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Pipeline'),
      '#description' => $this->t('The pipeline ID. Leave empty to use the default pipeline.'),
      '#default_value' => $this->configuration['pipeline'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['amount'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Amount'),
      '#description' => $this->t('The deal amount (numeric value).'),
      '#default_value' => $this->configuration['amount'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['closedate'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Close Date'),
      '#description' => $this->t('The expected close date in ISO 8601 format (YYYY-MM-DD) or Unix timestamp.'),
      '#default_value' => $this->configuration['closedate'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['associated_contact_ids'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Associated Contact IDs'),
      '#description' => $this->t('Comma-separated list of contact IDs to associate with this deal.'),
      '#default_value' => $this->configuration['associated_contact_ids'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['associated_company_ids'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Associated Company IDs'),
      '#description' => $this->t('Comma-separated list of company IDs to associate with this deal.'),
      '#default_value' => $this->configuration['associated_company_ids'],
      '#eca_token_replacement' => TRUE,
    ];

    $form = $this->addYamlPropertiesField($form, $form_state, 'deal');
    $form = $this->addTokenOutputField($form, $form_state, $this->t('Provide the name of a token that holds the deal data including ID, stage, and all properties.'));

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $dealname = $this->tokenService->replacePlain($this->configuration['dealname']);
    if (empty($dealname)) {
      $this->logError('Deal name is required to create deal.');
      return;
    }

    $dealstage = $this->tokenService->replacePlain($this->configuration['dealstage']);
    if (empty($dealstage)) {
      $this->logError('Deal stage is required to create deal.');
      return;
    }

    // Build deal properties.
    $properties = [
      'dealname' => $dealname,
      'dealstage' => $dealstage,
    ];

    // Add optional properties if provided.
    $pipeline = $this->tokenService->replacePlain($this->configuration['pipeline']);
    if (!empty($pipeline)) {
      $properties['pipeline'] = $pipeline;
    }

    $amount = $this->tokenService->replacePlain($this->configuration['amount']);
    if (!empty($amount)) {
      $properties['amount'] = $amount;
    }

    $closedate = $this->tokenService->replacePlain($this->configuration['closedate']);
    if (!empty($closedate)) {
      // Use the HubSpot service to format the date properly.
      $formatted_closedate = $this->hubspotService->formatDateTimeForApi($closedate);
      if ($formatted_closedate !== NULL) {
        $properties['closedate'] = $formatted_closedate;
      }
    }

    // Merge additional properties from YAML.
    $properties = $this->mergeYamlSettings($properties, 'additional_properties');

    // Build associations array.
    $associations = [];

    $contact_ids = $this->tokenService->replacePlain($this->configuration['associated_contact_ids']);
    if (!empty($contact_ids)) {
      $associations['contact'] = array_map('trim', explode(',', $contact_ids));
    }

    $company_ids = $this->tokenService->replacePlain($this->configuration['associated_company_ids']);
    if (!empty($company_ids)) {
      $associations['company'] = array_map('trim', explode(',', $company_ids));
    }

    // Create new deal.
    $response = $this->hubspotService->createDeal($properties, $associations);

    // Store response in token if configured.
    $this->storeResponseToken($response);
  }

}
