<?php

declare(strict_types=1);

namespace Drupal\eca_hubspot\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;

/**
 * Create a new HubSpot note.
 */
#[Action(
  id: 'eca_hubspot_create_note',
  label: new TranslatableMarkup('HubSpot: Create Note'),
  category: new TranslatableMarkup('HubSpot'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Create a new note in HubSpot.'),
  version_introduced: '1.0.0',
)]
final class CreateNote extends HubSpotActionBase {

  public function defaultConfiguration(): array {
    return [
      'content' => '',
      'additional_properties' => '',
      'associated_contact_ids' => '',
      'associated_company_ids' => '',
      'associated_deal_ids' => '',
      'token_name' => '',
    ] + parent::defaultConfiguration();
  }

  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['content'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Note Content'),
      '#description' => $this->t('The note body content (required).'),
      '#default_value' => $this->configuration['content'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form = $this->addYamlPropertiesField($form, $form_state, 'note');

    $form['associated_contact_ids'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Associated Contact IDs'),
      '#description' => $this->t('Comma-separated list of contact IDs to associate with this note.'),
      '#default_value' => $this->configuration['associated_contact_ids'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['associated_company_ids'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Associated Company IDs'),
      '#description' => $this->t('Comma-separated list of company IDs.'),
      '#default_value' => $this->configuration['associated_company_ids'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['associated_deal_ids'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Associated Deal IDs'),
      '#description' => $this->t('Comma-separated list of deal IDs.'),
      '#default_value' => $this->configuration['associated_deal_ids'],
      '#eca_token_replacement' => TRUE,
    ];

    $form = $this->addTokenOutputField($form, $form_state, $this->t('Token that holds the created note data.'));

    return $form;
  }

  public function execute(): void {
    $content = $this->tokenService->replace($this->configuration['content']);
    if (empty($content)) {
      $this->logError('Content is required to create note.');
      return;
    }

    // Notes support HTML, so keep it.
    $content = trim($content);

    $properties = ['hs_note_body' => $content];

    // Merge additional properties from YAML.
    $properties = $this->mergeYamlSettings($properties, 'additional_properties');

    $associations = [];

    $contact_ids = $this->tokenService->replacePlain($this->configuration['associated_contact_ids']);
    if (!empty($contact_ids)) {
      $associations['contact'] = array_map('trim', explode(',', $contact_ids));
    }

    $company_ids = $this->tokenService->replacePlain($this->configuration['associated_company_ids']);
    if (!empty($company_ids)) {
      $associations['company'] = array_map('trim', explode(',', $company_ids));
    }

    $deal_ids = $this->tokenService->replacePlain($this->configuration['associated_deal_ids']);
    if (!empty($deal_ids)) {
      $associations['deal'] = array_map('trim', explode(',', $deal_ids));
    }

    $response = $this->hubspotService->createNote($properties, $associations);
    $this->storeResponseToken($response);
  }

}
