<?php

declare(strict_types=1);

namespace Drupal\eca_hubspot\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;

/**
 * Create a new HubSpot task.
 */
#[Action(
  id: 'eca_hubspot_create_task',
  label: new TranslatableMarkup('HubSpot: Create Task'),
  category: new TranslatableMarkup('HubSpot'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Create a new task in HubSpot.'),
  version_introduced: '1.0.0',
)]
final class CreateTask extends HubSpotActionBase {

  public function defaultConfiguration(): array {
    return [
      'subject' => '',
      'body' => '',
      'due_date' => '',
      'priority' => '',
      'task_type' => '',
      'owner_id' => '',
      'additional_properties' => '',
      'associated_contact_ids' => '',
      'associated_company_ids' => '',
      'associated_deal_ids' => '',
      'token_name' => '',
    ] + parent::defaultConfiguration();
  }

  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['subject'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Subject'),
      '#description' => $this->t('The task title/subject (required).'),
      '#default_value' => $this->configuration['subject'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['body'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Body'),
      '#description' => $this->t('The task description.'),
      '#default_value' => $this->configuration['body'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['due_date'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Due Date'),
      '#description' => $this->t('Due date in ISO 8601 format or Unix timestamp.'),
      '#default_value' => $this->configuration['due_date'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['priority'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Priority'),
      '#description' => $this->t('Task priority (e.g., "LOW", "MEDIUM", "HIGH").'),
      '#default_value' => $this->configuration['priority'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['task_type'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Task Type'),
      '#description' => $this->t('The task type (e.g., "TODO", "EMAIL", "CALL").'),
      '#default_value' => $this->configuration['task_type'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['owner_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Owner ID'),
      '#description' => $this->t('HubSpot user ID to assign this task to.'),
      '#default_value' => $this->configuration['owner_id'],
      '#eca_token_replacement' => TRUE,
    ];

    $form = $this->addYamlPropertiesField($form, $form_state, 'task');

    $form['associated_contact_ids'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Associated Contact IDs'),
      '#description' => $this->t('Comma-separated list of contact IDs.'),
      '#default_value' => $this->configuration['associated_contact_ids'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['associated_company_ids'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Associated Company IDs'),
      '#description' => $this->t('Comma-separated list of company IDs.'),
      '#default_value' => $this->configuration['associated_company_ids'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['associated_deal_ids'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Associated Deal IDs'),
      '#description' => $this->t('Comma-separated list of deal IDs.'),
      '#default_value' => $this->configuration['associated_deal_ids'],
      '#eca_token_replacement' => TRUE,
    ];

    $form = $this->addTokenOutputField($form, $form_state, $this->t('Token that holds the created task data.'));

    return $form;
  }

  public function execute(): void {
    $task_data = [];

    $subject = $this->tokenService->replaceClear($this->configuration['subject']);
    if (empty($subject)) {
      $this->logError('Subject is required to create task.');
      return;
    }
    $task_data['hs_task_subject'] = $subject;

    $body = $this->tokenService->replace($this->configuration['body']);
    if (!empty($body)) {
      // Tasks support HTML, so keep it.
      $body = trim($body);
      $task_data['hs_task_body'] = $body;
    }

    $due_date = $this->tokenService->replacePlain($this->configuration['due_date']);
    if (!empty($due_date)) {
      $formatted_date = $this->hubspotService->formatDateTimeForApi($due_date);
      if ($formatted_date !== NULL) {
        $task_data['hs_timestamp'] = $formatted_date;
      }
    }

    $priority = $this->tokenService->replacePlain($this->configuration['priority']);
    if (!empty($priority)) {
      $task_data['hs_task_priority'] = $priority;
    }

    $task_type = $this->tokenService->replacePlain($this->configuration['task_type']);
    if (!empty($task_type)) {
      $task_data['hs_task_type'] = $task_type;
    }

    $owner_id = $this->tokenService->replacePlain($this->configuration['owner_id']);
    if (!empty($owner_id)) {
      $task_data['hubspot_owner_id'] = $owner_id;
    }

    // Merge additional properties from YAML.
    $task_data = $this->mergeYamlSettings($task_data, 'additional_properties');

    $associations = [];

    $contact_ids = $this->tokenService->replacePlain($this->configuration['associated_contact_ids']);
    if (!empty($contact_ids)) {
      $associations['contact'] = array_map('trim', explode(',', $contact_ids));
    }

    $company_ids = $this->tokenService->replacePlain($this->configuration['associated_company_ids']);
    if (!empty($company_ids)) {
      $associations['company'] = array_map('trim', explode(',', $company_ids));
    }

    $deal_ids = $this->tokenService->replacePlain($this->configuration['associated_deal_ids']);
    if (!empty($deal_ids)) {
      $associations['deal'] = array_map('trim', explode(',', $deal_ids));
    }

    $response = $this->hubspotService->createTask($task_data, $associations);
    $this->storeResponseToken($response);
  }

}
