<?php

declare(strict_types=1);

namespace Drupal\eca_hubspot\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;

/**
 * Create a new HubSpot ticket.
 */
#[Action(
  id: 'eca_hubspot_create_ticket',
  label: new TranslatableMarkup('HubSpot: Create Ticket'),
  category: new TranslatableMarkup('HubSpot'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Create a new ticket record in HubSpot.'),
  version_introduced: '1.0.0',
)]
final class CreateTicket extends HubSpotActionBase {

  public function defaultConfiguration(): array {
    return [
      'subject' => '',
      'content' => '',
      'pipeline' => '',
      'stage' => '',
      'priority' => '',
      'associated_contact_ids' => '',
      'associated_company_ids' => '',
      'associated_deal_ids' => '',
      'additional_properties' => '',
      'token_name' => '',
    ] + parent::defaultConfiguration();
  }

  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['subject'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Subject'),
      '#description' => $this->t('The ticket subject (required).'),
      '#default_value' => $this->configuration['subject'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['content'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Content'),
      '#description' => $this->t('The ticket content/description.'),
      '#default_value' => $this->configuration['content'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['pipeline'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Pipeline'),
      '#description' => $this->t('The pipeline ID (internal ID number). Leave empty to use default pipeline.'),
      '#default_value' => $this->configuration['pipeline'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['stage'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Stage'),
      '#description' => $this->t('The ticket stage/status (internal ID number, required).'),
      '#default_value' => $this->configuration['stage'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['priority'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Priority'),
      '#description' => $this->t('The priority (e.g., "LOW", "MEDIUM", "HIGH").'),
      '#default_value' => $this->configuration['priority'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['associated_contact_ids'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Associated Contact IDs'),
      '#description' => $this->t('Comma-separated list of contact IDs.'),
      '#default_value' => $this->configuration['associated_contact_ids'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['associated_company_ids'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Associated Company IDs'),
      '#description' => $this->t('Comma-separated list of company IDs.'),
      '#default_value' => $this->configuration['associated_company_ids'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['associated_deal_ids'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Associated Deal IDs'),
      '#description' => $this->t('Comma-separated list of deal IDs.'),
      '#default_value' => $this->configuration['associated_deal_ids'],
      '#eca_token_replacement' => TRUE,
    ];

    $form = $this->addYamlPropertiesField($form, $form_state, 'ticket');
    $form = $this->addTokenOutputField($form, $form_state, $this->t('Token that holds the created ticket data.'));

    return $form;
  }

  public function execute(): void {
    $properties = [];

    $subject = $this->tokenService->replaceClear($this->configuration['subject']);
    if (empty($subject)) {
      $this->logError('Subject is required to create ticket.');
      return;
    }
    $properties['subject'] = $subject;

    $stage = $this->tokenService->replacePlain($this->configuration['stage']);
    if (empty($stage)) {
      $this->logError('Stage is required to create ticket.');
      return;
    }
    $properties['hs_pipeline_stage'] = $stage;

    $content = $this->tokenService->replace($this->configuration['content']);
    if (!empty($content)) {
      // Convert <p> and <br> to line breaks, then strip remaining HTML.
      $content = str_replace(['</p>', '<br>', '<br/>', '<br />'], "\n", $content);
      $content = strip_tags($content);
      $content = trim($content);
      $properties['content'] = $content;
    }

    $pipeline = $this->tokenService->replacePlain($this->configuration['pipeline']);
    if (!empty($pipeline)) {
      $properties['hs_pipeline'] = $pipeline;
    }

    $priority = $this->tokenService->replacePlain($this->configuration['priority']);
    if (!empty($priority)) {
      $properties['hs_ticket_priority'] = $priority;
    }

    $properties = $this->mergeYamlSettings($properties, 'additional_properties');

    $associations = [];

    $contact_ids = $this->tokenService->replacePlain($this->configuration['associated_contact_ids']);
    if (!empty($contact_ids)) {
      $associations['contact'] = array_map('trim', explode(',', $contact_ids));
    }

    $company_ids = $this->tokenService->replacePlain($this->configuration['associated_company_ids']);
    if (!empty($company_ids)) {
      $associations['company'] = array_map('trim', explode(',', $company_ids));
    }

    $deal_ids = $this->tokenService->replacePlain($this->configuration['associated_deal_ids']);
    if (!empty($deal_ids)) {
      $associations['deal'] = array_map('trim', explode(',', $deal_ids));
    }

    $response = $this->hubspotService->createTicket($properties, $associations);
    $this->storeResponseToken($response);
  }

}
