<?php

declare(strict_types=1);

namespace Drupal\eca_hubspot\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;

/**
 * Delete a HubSpot contact.
 */
#[Action(
  id: 'eca_hubspot_delete_contact',
  label: new TranslatableMarkup('HubSpot: Delete Contact'),
  category: new TranslatableMarkup('HubSpot'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Delete a contact record from HubSpot by ID.'),
  version_introduced: '1.0.0',
)]
final class DeleteContact extends HubSpotActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'contact_id' => '',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['contact_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Contact ID'),
      '#description' => $this->t('The HubSpot contact ID to delete.'),
      '#default_value' => $this->configuration['contact_id'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $contact_id = $this->tokenService->replacePlain($this->configuration['contact_id']);
    if (empty($contact_id)) {
      $this->logError('Contact ID is required to delete contact.');
      return;
    }

    // Delete the contact.
    $success = $this->hubspotService->deleteContact($contact_id);

    if (!$success) {
      $this->logError('Failed to delete contact with ID: @id', ['@id' => $contact_id]);
    }
  }

}
