<?php

declare(strict_types=1);

namespace Drupal\eca_hubspot\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;

/**
 * Delete a HubSpot deal.
 */
#[Action(
  id: 'eca_hubspot_delete_deal',
  label: new TranslatableMarkup('HubSpot: Delete Deal'),
  category: new TranslatableMarkup('HubSpot'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Delete a deal record from HubSpot by ID.'),
  version_introduced: '1.0.0',
)]
final class DeleteDeal extends HubSpotActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'deal_id' => '',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['deal_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Deal ID'),
      '#description' => $this->t('The HubSpot deal ID to delete.'),
      '#default_value' => $this->configuration['deal_id'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $deal_id = $this->tokenService->replacePlain($this->configuration['deal_id']);
    if (empty($deal_id)) {
      $this->logError('Deal ID is required to delete deal.');
      return;
    }

    // Delete the deal.
    $success = $this->hubspotService->deleteDeal($deal_id);

    if (!$success) {
      $this->logError('Failed to delete deal with ID: @id', ['@id' => $deal_id]);
    }
  }

}
