<?php

declare(strict_types=1);

namespace Drupal\eca_hubspot\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;

/**
 * Delete a HubSpot lead.
 */
#[Action(
  id: 'eca_hubspot_delete_lead',
  label: new TranslatableMarkup('HubSpot: Delete Lead'),
  category: new TranslatableMarkup('HubSpot'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Delete a lead record from HubSpot by ID.'),
  version_introduced: '1.0.0',
)]
final class DeleteLead extends HubSpotActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'lead_id' => '',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['lead_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Lead ID'),
      '#description' => $this->t('The HubSpot lead ID to delete.'),
      '#default_value' => $this->configuration['lead_id'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $lead_id = $this->tokenService->replacePlain($this->configuration['lead_id']);
    if (empty($lead_id)) {
      $this->logError('Lead ID is required to delete lead.');
      return;
    }

    // Delete the lead.
    $success = $this->hubspotService->deleteLead($lead_id);

    if (!$success) {
      $this->logError('Failed to delete lead with ID: @id', ['@id' => $lead_id]);
    }
  }

}
