<?php

declare(strict_types=1);

namespace Drupal\eca_hubspot\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;

/**
 * Delete a HubSpot ticket.
 */
#[Action(
  id: 'eca_hubspot_delete_ticket',
  label: new TranslatableMarkup('HubSpot: Delete Ticket'),
  category: new TranslatableMarkup('HubSpot'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Delete a ticket record from HubSpot by ID.'),
  version_introduced: '1.0.0',
)]
final class DeleteTicket extends HubSpotActionBase {

  public function defaultConfiguration(): array {
    return [
      'ticket_id' => '',
    ] + parent::defaultConfiguration();
  }

  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['ticket_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Ticket ID'),
      '#description' => $this->t('The HubSpot ticket ID to delete.'),
      '#default_value' => $this->configuration['ticket_id'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    return $form;
  }

  public function execute(): void {
    $ticket_id = $this->tokenService->replacePlain($this->configuration['ticket_id']);
    if (empty($ticket_id)) {
      $this->logError('Ticket ID is required to delete ticket.');
      return;
    }

    $success = $this->hubspotService->deleteTicket($ticket_id);

    if (!$success) {
      $this->logError('Failed to delete ticket with ID: @id', ['@id' => $ticket_id]);
    }
  }

}
