<?php

declare(strict_types=1);

namespace Drupal\eca_hubspot\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;

/**
 * Get associations for a HubSpot object.
 */
#[Action(
  id: 'eca_hubspot_get_associations',
  label: new TranslatableMarkup('HubSpot: Get Associations'),
  category: new TranslatableMarkup('HubSpot'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Retrieve all associations for an object in HubSpot.'),
  version_introduced: '1.0.0',
)]
final class GetAssociations extends HubSpotActionBase {

  public function defaultConfiguration(): array {
    return [
      'object_type' => '',
      'object_id' => '',
      'to_object_type' => '',
      'token_name' => '',
    ] + parent::defaultConfiguration();
  }

  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['object_type'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Object Type'),
      '#description' => $this->t('The object type (e.g., "contact", "company", "deal", "ticket", "lead").'),
      '#default_value' => $this->configuration['object_type'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['object_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Object ID'),
      '#description' => $this->t('The object ID.'),
      '#default_value' => $this->configuration['object_id'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['to_object_type'] = [
      '#type' => 'textfield',
      '#title' => $this->t('To Object Type'),
      '#description' => $this->t('The type of associated objects to retrieve (e.g., "contact", "company", "deal", "ticket", "lead").'),
      '#default_value' => $this->configuration['to_object_type'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form = $this->addTokenOutputField($form, $form_state, $this->t('Token that holds the array of associated object IDs.'));

    return $form;
  }

  public function execute(): void {
    $object_type = $this->tokenService->replacePlain($this->configuration['object_type']);
    $object_id = $this->tokenService->replacePlain($this->configuration['object_id']);
    $to_object_type = $this->tokenService->replacePlain($this->configuration['to_object_type']);

    if (empty($object_type) || empty($object_id) || empty($to_object_type)) {
      $this->logError('All fields are required to get associations.');
      return;
    }

    $response = $this->hubspotService->getAssociations(
      $object_type,
      $object_id,
      $to_object_type
    );

    $this->storeResponseToken($response);
  }

}
