<?php

declare(strict_types=1);

namespace Drupal\eca_hubspot\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;

/**
 * Get a HubSpot company.
 */
#[Action(
  id: 'eca_hubspot_get_company',
  label: new TranslatableMarkup('HubSpot: Get Company'),
  category: new TranslatableMarkup('HubSpot'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Retrieve company by ID or domain for workflow decisions.'),
  version_introduced: '1.0.0',
)]
final class GetCompany extends HubSpotActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'company_id_or_domain' => '',
      'properties' => '',
      'token_name' => '',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['company_id_or_domain'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Company ID or Domain'),
      '#description' => $this->t('The HubSpot company ID or domain to retrieve.'),
      '#default_value' => $this->configuration['company_id_or_domain'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['properties'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Properties to Retrieve'),
      '#description' => $this->t('Comma-separated list of property names to retrieve. Leave empty to retrieve the default properties. Example: name,domain,industry,phone'),
      '#default_value' => $this->configuration['properties'],
      '#eca_token_replacement' => TRUE,
    ];

    $form = $this->addTokenOutputField($form, $form_state, $this->t('Provide the name of a token that holds the company data including all requested properties.'));

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $company_id_or_domain = $this->tokenService->replacePlain($this->configuration['company_id_or_domain']);
    if (empty($company_id_or_domain)) {
      $this->logError('Company ID or Domain is required to get company.');
      return;
    }

    // Parse properties list.
    $properties = [];
    $properties_string = $this->tokenService->replacePlain($this->configuration['properties']);
    if (!empty($properties_string)) {
      $properties = array_map('trim', explode(',', $properties_string));
    }

    // Get the company.
    $response = $this->hubspotService->getCompany($company_id_or_domain, $properties);
    // Store response in token if configured.
    $this->storeResponseToken($response);
  }

}
