<?php

declare(strict_types=1);

namespace Drupal\eca_hubspot\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;

/**
 * Get a HubSpot deal.
 */
#[Action(
  id: 'eca_hubspot_get_deal',
  label: new TranslatableMarkup('HubSpot: Get Deal'),
  category: new TranslatableMarkup('HubSpot'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Retrieve deal details by ID for workflow decisions.'),
  version_introduced: '1.0.0',
)]
final class GetDeal extends HubSpotActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'deal_id' => '',
      'properties' => '',
      'include_associations' => FALSE,
      'token_name' => '',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['deal_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Deal ID'),
      '#description' => $this->t('The HubSpot deal ID to retrieve.'),
      '#default_value' => $this->configuration['deal_id'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['properties'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Properties to Retrieve'),
      '#description' => $this->t('Comma-separated list of property names to retrieve. Leave empty to retrieve the default properties. Example: dealname,dealstage,amount,closedate'),
      '#default_value' => $this->configuration['properties'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['include_associations'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Include Associations'),
      '#description' => $this->t('Include associated contacts and companies in the response.'),
      '#default_value' => $this->configuration['include_associations'],
    ];

    $form = $this->addTokenOutputField($form, $form_state, $this->t('Provide the name of a token that holds the deal data including all requested properties and associations.'));

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $deal_id = $this->tokenService->replacePlain($this->configuration['deal_id']);
    if (empty($deal_id)) {
      $this->logError('Deal ID is required to get deal.');
      return;
    }

    // Parse properties list.
    $properties = [];
    $properties_string = $this->tokenService->replacePlain($this->configuration['properties']);
    if (!empty($properties_string)) {
      $properties = array_map('trim', explode(',', $properties_string));
    }

    // Get include associations setting.
    $include_associations = !empty($this->configuration['include_associations']);

    // Get the deal.
    $response = $this->hubspotService->getDeal($deal_id, $properties, $include_associations);
    // Store response in token if configured.
    $this->storeResponseToken($response);
  }

}
