<?php

declare(strict_types=1);

namespace Drupal\eca_hubspot\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;

/**
 * Get a HubSpot note.
 */
#[Action(
  id: 'eca_hubspot_get_note',
  label: new TranslatableMarkup('HubSpot: Get Note'),
  category: new TranslatableMarkup('HubSpot'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Retrieve a note record from HubSpot by ID.'),
  version_introduced: '1.0.0',
)]
final class GetNote extends HubSpotActionBase {

  public function defaultConfiguration(): array {
    return [
      'note_id' => '',
      'properties' => '',
      'token_name' => '',
    ] + parent::defaultConfiguration();
  }

  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['note_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Note ID'),
      '#description' => $this->t('The HubSpot note ID to retrieve.'),
      '#default_value' => $this->configuration['note_id'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['properties'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Properties to Retrieve'),
      '#description' => $this->t('Comma-separated list of property names. Leave empty for all.'),
      '#default_value' => $this->configuration['properties'],
      '#eca_token_replacement' => TRUE,
    ];

    $form = $this->addTokenOutputField($form, $form_state, $this->t('Token that holds the retrieved note data.'));

    return $form;
  }

  public function execute(): void {
    $note_id = $this->tokenService->replacePlain($this->configuration['note_id']);
    if (empty($note_id)) {
      $this->logError('Note ID is required to get note.');
      return;
    }

    $properties = [];
    $properties_list = $this->tokenService->replacePlain($this->configuration['properties']);
    if (!empty($properties_list)) {
      $properties = array_map('trim', explode(',', $properties_list));
    }

    $response = $this->hubspotService->getNote($note_id, $properties);
    $this->storeResponseToken($response);
  }

}
