<?php

declare(strict_types=1);

namespace Drupal\eca_hubspot\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;

/**
 * Get a HubSpot pipeline.
 */
#[Action(
  id: 'eca_hubspot_get_pipeline',
  label: new TranslatableMarkup('HubSpot: Get Pipeline'),
  category: new TranslatableMarkup('HubSpot'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Get a specific pipeline by ID with all its stages.'),
  version_introduced: '1.0.0',
)]
final class GetPipeline extends HubSpotActionBase {

  public function defaultConfiguration(): array {
    return [
      'object_type' => 'deals',
      'object_type_custom' => '',
      'pipeline_id' => '',
      'token_name' => '',
    ] + parent::defaultConfiguration();
  }

  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['object_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Object Type'),
      '#description' => $this->t('The type of object this pipeline belongs to.'),
      '#options' => [
        'deals' => $this->t('Deals'),
        'tickets' => $this->t('Tickets'),
        'leads' => $this->t('Leads'),
        'appointments' => $this->t('Appointments'),
        'courses' => $this->t('Courses'),
        'listings' => $this->t('Listings'),
        'orders' => $this->t('Orders'),
        'services' => $this->t('Services'),
      ],
      '#default_value' => $this->configuration['object_type'],
      '#required' => TRUE,
    ];

    $form['object_type_custom'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Custom Object Type'),
      '#description' => $this->t('If using a custom object type, enter it here. This will override the selection above.'),
      '#default_value' => $this->configuration['object_type_custom'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['pipeline_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Pipeline ID'),
      '#description' => $this->t('The pipeline ID to retrieve.'),
      '#default_value' => $this->configuration['pipeline_id'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form = $this->addTokenOutputField($form, $form_state, $this->t('Token that holds the pipeline data including: id, label, displayOrder, and stages array (each stage has id, label, displayOrder, metadata).'));

    return $form;
  }

  public function execute(): void {
    $custom_type = $this->tokenService->replacePlain($this->configuration['object_type_custom']);
    $object_type = !empty($custom_type) ? $custom_type : $this->configuration['object_type'];

    $pipeline_id = $this->tokenService->replacePlain($this->configuration['pipeline_id']);

    if (empty($object_type) || empty($pipeline_id)) {
      $this->logError('Object type and pipeline ID are required.');
      return;
    }

    $response = $this->hubspotService->getPipeline($object_type, $pipeline_id);
    $this->storeResponseToken($response);
  }

}
