<?php

declare(strict_types=1);

namespace Drupal\eca_hubspot\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;

/**
 * Get a HubSpot task.
 */
#[Action(
  id: 'eca_hubspot_get_task',
  label: new TranslatableMarkup('HubSpot: Get Task'),
  category: new TranslatableMarkup('HubSpot'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Retrieve a task record from HubSpot by ID.'),
  version_introduced: '1.0.0',
)]
final class GetTask extends HubSpotActionBase {

  public function defaultConfiguration(): array {
    return [
      'task_id' => '',
      'properties' => '',
      'token_name' => '',
    ] + parent::defaultConfiguration();
  }

  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['task_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Task ID'),
      '#description' => $this->t('The HubSpot task ID to retrieve.'),
      '#default_value' => $this->configuration['task_id'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['properties'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Properties to Retrieve'),
      '#description' => $this->t('Comma-separated list of property names. Leave empty for all.'),
      '#default_value' => $this->configuration['properties'],
      '#eca_token_replacement' => TRUE,
    ];

    $form = $this->addTokenOutputField($form, $form_state, $this->t('Token that holds the retrieved task data.'));

    return $form;
  }

  public function execute(): void {
    $task_id = $this->tokenService->replacePlain($this->configuration['task_id']);
    if (empty($task_id)) {
      $this->logError('Task ID is required to get task.');
      return;
    }

    $properties = [];
    $properties_list = $this->tokenService->replacePlain($this->configuration['properties']);
    if (!empty($properties_list)) {
      $properties = array_map('trim', explode(',', $properties_list));
    }

    $response = $this->hubspotService->getTask($task_id, $properties);
    $this->storeResponseToken($response);
  }

}
