<?php

declare(strict_types=1);

namespace Drupal\eca_hubspot\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;

/**
 * Update a HubSpot deal.
 */
#[Action(
  id: 'eca_hubspot_update_deal',
  label: new TranslatableMarkup('HubSpot: Update Deal'),
  category: new TranslatableMarkup('HubSpot'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Update an existing deal in HubSpot by ID.'),
  version_introduced: '1.0.0',
)]
final class UpdateDeal extends HubSpotActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'deal_id' => '',
      'dealname' => '',
      'dealstage' => '',
      'pipeline' => '',
      'amount' => '',
      'closedate' => '',
      'associated_contact_ids' => '',
      'associated_company_ids' => '',
      'additional_properties' => '',
      'token_name' => '',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['deal_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Deal ID'),
      '#description' => $this->t('The HubSpot deal ID to update.'),
      '#default_value' => $this->configuration['deal_id'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['dealname'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Deal Name'),
      '#description' => $this->t('The name of the deal.'),
      '#default_value' => $this->configuration['dealname'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['dealstage'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Deal Stage'),
      '#description' => $this->t('The deal stage ID to move the deal to (e.g., "appointmentscheduled", "qualifiedtobuy", "closedwon").'),
      '#default_value' => $this->configuration['dealstage'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['pipeline'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Pipeline'),
      '#description' => $this->t('The pipeline ID. Leave empty to keep current pipeline.'),
      '#default_value' => $this->configuration['pipeline'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['amount'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Amount'),
      '#description' => $this->t('The deal amount (numeric value).'),
      '#default_value' => $this->configuration['amount'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['closedate'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Close Date'),
      '#description' => $this->t('The expected close date in ISO 8601 format (YYYY-MM-DD) or Unix timestamp.'),
      '#default_value' => $this->configuration['closedate'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['associated_contact_ids'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Associated Contact IDs'),
      '#description' => $this->t('Comma-separated list of contact IDs to associate with this deal.'),
      '#default_value' => $this->configuration['associated_contact_ids'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['associated_company_ids'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Associated Company IDs'),
      '#description' => $this->t('Comma-separated list of company IDs to associate with this deal.'),
      '#default_value' => $this->configuration['associated_company_ids'],
      '#eca_token_replacement' => TRUE,
    ];

    $form = $this->addYamlPropertiesField($form, $form_state, 'deal');
    $form = $this->addTokenOutputField($form, $form_state, $this->t('Provide the name of a token that holds the updated deal data including ID, stage, and all properties.'));

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $deal_id = $this->tokenService->replacePlain($this->configuration['deal_id']);
    if (empty($deal_id)) {
      $this->logError('Deal ID is required to update deal.');
      return;
    }

    // Build deal properties to update.
    $properties = [];

    // Add properties if provided.
    $dealname = $this->tokenService->replacePlain($this->configuration['dealname']);
    if (!empty($dealname)) {
      $properties['dealname'] = $dealname;
    }

    $dealstage = $this->tokenService->replacePlain($this->configuration['dealstage']);
    if (!empty($dealstage)) {
      $properties['dealstage'] = $dealstage;
    }

    // Add optional properties if provided.
    $pipeline = $this->tokenService->replacePlain($this->configuration['pipeline']);
    if (!empty($pipeline)) {
      $properties['pipeline'] = $pipeline;
    }

    $amount = $this->tokenService->replacePlain($this->configuration['amount']);
    if (!empty($amount)) {
      $properties['amount'] = $amount;
    }

    $closedate = $this->tokenService->replacePlain($this->configuration['closedate']);
    if (!empty($closedate)) {
      // Use the HubSpot service to format the date properly.
      $formatted_closedate = $this->hubspotService->formatDateTimeForApi($closedate);
      if ($formatted_closedate !== NULL) {
        $properties['closedate'] = $formatted_closedate;
      }
    }

    // Merge additional properties from YAML.
    $properties = $this->mergeYamlSettings($properties, 'additional_properties');

    // Build associations array.
    $associations = [];

    $contact_ids = $this->tokenService->replacePlain($this->configuration['associated_contact_ids']);
    if (!empty($contact_ids)) {
      $associations['contact'] = array_map('trim', explode(',', $contact_ids));
    }

    $company_ids = $this->tokenService->replacePlain($this->configuration['associated_company_ids']);
    if (!empty($company_ids)) {
      $associations['company'] = array_map('trim', explode(',', $company_ids));
    }

    // Update the deal.
    $response = $this->hubspotService->updateDeal($deal_id, $properties, $associations);

    // Store response in token if configured.
    $this->storeResponseToken($response);
  }

}
