<?php

declare(strict_types=1);

namespace Drupal\eca_hubspot\Service;

use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Utility\Token;
use Drupal\eca_hubspot\DateTimeFormattingTrait;
use Drupal\hubspot_api\ManagerInterface;
use HubSpot\Discovery\Discovery;

/**
 * Service for HubSpot CRM API operations.
 */
class HubSpotService {

  use DateTimeFormattingTrait;

  /**
   * Object type: Contact.
   */
  const string OBJECT_TYPE_CONTACT = 'contact';

  /**
   * Object type: Company.
   */
  const string OBJECT_TYPE_COMPANY = 'company';

  /**
   * Object type: Deal.
   */
  const string OBJECT_TYPE_DEAL = 'deal';

  /**
   * Object type: Lead.
   */
  const string OBJECT_TYPE_LEAD = 'lead';

  /**
   * Object type: Ticket.
   */
  const string OBJECT_TYPE_TICKET = 'ticket';

  /**
   * Object type: Note.
   */
  const string OBJECT_TYPE_NOTE = 'note';

  /**
   * Object type: Task.
   */
  const string OBJECT_TYPE_TASK = 'task';

  /**
   * Contact lifecycle stage: Subscriber.
   */
  const string LIFECYCLE_SUBSCRIBER = 'subscriber';

  /**
   * Contact lifecycle stage: Lead.
   */
  const string LIFECYCLE_LEAD = 'lead';

  /**
   * Contact lifecycle stage: Marketing Qualified Lead.
   */
  const string LIFECYCLE_MQL = 'marketingqualifiedlead';

  /**
   * Contact lifecycle stage: Sales Qualified Lead.
   */
  const string LIFECYCLE_SQL = 'salesqualifiedlead';

  /**
   * Contact lifecycle stage: Opportunity.
   */
  const string LIFECYCLE_OPPORTUNITY = 'opportunity';

  /**
   * Contact lifecycle stage: Customer.
   */
  const string LIFECYCLE_CUSTOMER = 'customer';

  /**
   * Contact lifecycle stage: Evangelist.
   */
  const string LIFECYCLE_EVANGELIST = 'evangelist';

  /**
   * Contact lifecycle stage: Other.
   */
  const string LIFECYCLE_OTHER = 'other';

  /**
   * Special value: Use token for dynamic value.
   */
  const string VALUE_TOKEN = 'token';

  /**
   * Ticket priority: Low.
   */
  const string PRIORITY_LOW = 'LOW';

  /**
   * Ticket priority: Medium.
   */
  const string PRIORITY_MEDIUM = 'MEDIUM';

  /**
   * Ticket priority: High.
   */
  const string PRIORITY_HIGH = 'HIGH';

  /**
   * Task type: TODO.
   */
  const string TASK_TYPE_TODO = 'TODO';

  /**
   * Task type: Email.
   */
  const string TASK_TYPE_EMAIL = 'EMAIL';

  /**
   * Task type: Call.
   */
  const string TASK_TYPE_CALL = 'CALL';

  /**
   * Task priority: None.
   */
  const string TASK_PRIORITY_NONE = 'NONE';

  /**
   * Task priority: Low.
   */
  const string TASK_PRIORITY_LOW = 'LOW';

  /**
   * Task priority: Medium.
   */
  const string TASK_PRIORITY_MEDIUM = 'MEDIUM';

  /**
   * Task priority: High.
   */
  const string TASK_PRIORITY_HIGH = 'HIGH';

  /**
   * The HubSpot API manager.
   */
  protected ManagerInterface $hubspotManager;

  /**
   * The logger channel.
   */
  protected LoggerChannelInterface $logger;

  /**
   * The token service.
   */
  protected Token $token;

  /**
   * Constructs a HubSpotService object.
   *
   * @param \Drupal\hubspot_api\ManagerInterface $hubspot_manager
   *   The HubSpot API manager.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger channel factory.
   * @param \Drupal\Core\Utility\Token $token
   *   The token service.
   */
  public function __construct(ManagerInterface $hubspot_manager, LoggerChannelFactoryInterface $logger_factory, Token $token) {
    $this->hubspotManager = $hubspot_manager;
    $this->logger = $logger_factory->get('eca_hubspot');
    $this->token = $token;
  }

  /**
   * Gets the HubSpot Discovery client.
   *
   * @return \HubSpot\Discovery\Discovery|null
   *   The HubSpot client or NULL on failure.
   */
  public function getClient(): ?Discovery {
    try {
      return $this->hubspotManager->getHandler();
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to get HubSpot client: @message', [
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Creates a contact in HubSpot.
   *
   * @param array $properties
   *   The contact properties.
   *
   * @return array|null
   *   The created contact data or NULL on failure.
   */
  public function createContact(array $properties): ?array {
    try {
      $client = $this->getClient();
      if (!$client) {
        return NULL;
      }

      $contactInput = new \HubSpot\Client\Crm\Contacts\Model\SimplePublicObjectInputForCreate([
        'properties' => $properties,
      ]);

      $response = $client->crm()->contacts()->basicApi()->create($contactInput);

      return $this->formatObjectResponse($response);
    }
    catch (\Exception $e) {
      $this->handleApiException($e, 'create contact');
      return NULL;
    }
  }

  /**
   * Updates a contact in HubSpot.
   *
   * @param string $contact_id
   *   The contact ID.
   * @param array $properties
   *   The properties to update.
   *
   * @return array|null
   *   The updated contact data or NULL on failure.
   */
  public function updateContact(string $contact_id, array $properties): ?array {
    try {
      $client = $this->getClient();
      if (!$client) {
        return NULL;
      }

      $contactInput = new \HubSpot\Client\Crm\Contacts\Model\SimplePublicObjectInput([
        'properties' => $properties,
      ]);

      $response = $client->crm()->contacts()->basicApi()->update($contact_id, $contactInput);

      return $this->formatObjectResponse($response);
    }
    catch (\Exception $e) {
      $this->handleApiException($e, 'update contact');
      return NULL;
    }
  }

  /**
   * Gets a contact from HubSpot.
   *
   * @param string $contact_id_or_email
   *   The contact ID or email address.
   * @param array $properties
   *   The properties to retrieve.
   *
   * @return array|null
   *   The contact data or NULL on failure.
   */
  public function getContact(string $contact_id_or_email, array $properties = []): ?array {
    try {
      $client = $this->getClient();
      if (!$client) {
        return NULL;
      }

      // If it looks like an email, use the email lookup endpoint.
      if (filter_var($contact_id_or_email, FILTER_VALIDATE_EMAIL)) {
        try {
          // Use idProperty=email to look up by email address.
          $response = $client->crm()->contacts()->basicApi()->getById(
            $contact_id_or_email,
            $properties,
            NULL,
            NULL,
            FALSE,
            'email'
          );

          return $this->formatObjectResponse($response);
        }
        catch (\HubSpot\Client\Crm\Contacts\ApiException $e) {
          // 404 means contact doesn't exist - this is expected, not an error.
          if ($e->getCode() === 404) {
            return NULL;
          }
          // Other errors should be logged.
          throw $e;
        }
      }

      // Otherwise get by ID.
      $response = $client->crm()->contacts()->basicApi()->getById(
        $contact_id_or_email,
        $properties,
        NULL,
        NULL,
        FALSE
      );

      return $this->formatObjectResponse($response);
    }
    catch (\Exception $e) {
      $this->handleApiException($e, 'get contact');
      return NULL;
    }
  }

  /**
   * Deletes a contact from HubSpot.
   *
   * @param string $contact_id
   *   The contact ID.
   *
   * @return bool
   *   TRUE on success, FALSE on failure.
   */
  public function deleteContact(string $contact_id): bool {
    try {
      $client = $this->getClient();
      if (!$client) {
        return FALSE;
      }

      $client->crm()->contacts()->basicApi()->archive($contact_id);

      return TRUE;
    }
    catch (\Exception $e) {
      $this->handleApiException($e, 'delete contact');
      return FALSE;
    }
  }

  /**
   * Searches for contacts in HubSpot.
   *
   * @param array $filters
   *   The search filters.
   * @param string $sort_by
   *   The property to sort by.
   * @param int $limit
   *   The maximum number of results.
   *
   * @return array|null
   *   The search results or NULL on failure.
   */
  public function searchContacts(array $filters, string $sort_by = '', int $limit = 100): ?array {
    return $this->performSearch('contacts', $filters, $sort_by, $limit);
  }

  /**
   * Creates a company in HubSpot.
   *
   * @param array $properties
   *   The company properties.
   *
   * @return array|null
   *   The created company data or NULL on failure.
   */
  public function createCompany(array $properties): ?array {
    try {
      $client = $this->getClient();
      if (!$client) {
        return NULL;
      }

      $companyInput = new \HubSpot\Client\Crm\Companies\Model\SimplePublicObjectInputForCreate([
        'properties' => $properties,
      ]);

      $response = $client->crm()->companies()->basicApi()->create($companyInput);

      return $this->formatObjectResponse($response);
    }
    catch (\Exception $e) {
      $this->handleApiException($e, 'create company');
      return NULL;
    }
  }

  /**
   * Updates a company in HubSpot.
   *
   * @param string $company_id
   *   The company ID.
   * @param array $properties
   *   The properties to update.
   *
   * @return array|null
   *   The updated company data or NULL on failure.
   */
  public function updateCompany(string $company_id, array $properties): ?array {
    try {
      $client = $this->getClient();
      if (!$client) {
        return NULL;
      }

      $companyInput = new \HubSpot\Client\Crm\Companies\Model\SimplePublicObjectInput([
        'properties' => $properties,
      ]);

      $response = $client->crm()->companies()->basicApi()->update($company_id, $companyInput);

      return $this->formatObjectResponse($response);
    }
    catch (\Exception $e) {
      $this->handleApiException($e, 'update company');
      return NULL;
    }
  }

  /**
   * Gets a company from HubSpot.
   *
   * @param string $company_id_or_domain
   *   The company ID or domain.
   * @param array $properties
   *   The properties to retrieve.
   *
   * @return array|null
   *   The company data or NULL on failure.
   */
  public function getCompany(string $company_id_or_domain, array $properties = []): ?array {
    try {
      $client = $this->getClient();
      if (!$client) {
        return NULL;
      }

      // If it looks like a domain, search by domain first.
      if (str_contains($company_id_or_domain, '.') && !is_numeric($company_id_or_domain)) {
        $searchResult = $this->searchCompanies([
          [
            'propertyName' => 'domain',
            'operator' => 'EQ',
            'value' => $company_id_or_domain,
          ],
        ], '', 1);

        if (!empty($searchResult['results'][0])) {
          return $searchResult['results'][0];
        }

        return NULL;
      }

      // Otherwise get by ID.
      $response = $client->crm()->companies()->basicApi()->getById(
        $company_id_or_domain,
        $properties,
        NULL,
        NULL,
        FALSE
      );

      return $this->formatObjectResponse($response);
    }
    catch (\Exception $e) {
      $this->handleApiException($e, 'get company');
      return NULL;
    }
  }

  /**
   * Deletes a company from HubSpot.
   *
   * @param string $company_id
   *   The company ID.
   *
   * @return bool
   *   TRUE on success, FALSE on failure.
   */
  public function deleteCompany(string $company_id): bool {
    try {
      $client = $this->getClient();
      if (!$client) {
        return FALSE;
      }

      $client->crm()->companies()->basicApi()->archive($company_id);

      return TRUE;
    }
    catch (\Exception $e) {
      $this->handleApiException($e, 'delete company');
      return FALSE;
    }
  }

  /**
   * Searches for companies in HubSpot.
   *
   * @param array $filters
   *   The search filters.
   * @param string $sort_by
   *   The property to sort by.
   * @param int $limit
   *   The maximum number of results.
   *
   * @return array|null
   *   The search results or NULL on failure.
   */
  public function searchCompanies(array $filters, string $sort_by = '', int $limit = 100): ?array {
    return $this->performSearch('companies', $filters, $sort_by, $limit);
  }

  /**
   * Creates a deal in HubSpot.
   *
   * @param array $properties
   *   The deal properties.
   * @param array $associations
   *   Optional associations to other objects.
   *
   * @return array|null
   *   The created deal data or NULL on failure.
   */
  public function createDeal(array $properties, array $associations = []): ?array {
    try {
      $client = $this->getClient();
      if (!$client) {
        return NULL;
      }

      $dealInput = new \HubSpot\Client\Crm\Deals\Model\SimplePublicObjectInputForCreate([
        'properties' => $properties,
        'associations' => $this->formatAssociations($associations, self::OBJECT_TYPE_DEAL),
      ]);

      $response = $client->crm()->deals()->basicApi()->create($dealInput);

      return $this->formatObjectResponse($response);
    }
    catch (\Exception $e) {
      $this->handleApiException($e, 'create deal');
      return NULL;
    }
  }

  /**
   * Updates a deal in HubSpot.
   *
   * @param string $deal_id
   *   The deal ID.
   * @param array $properties
   *   The properties to update.
   * @param array $associations
   *   Optional associations to add to the deal.
   *
   * @return array|null
   *   The updated deal data or NULL on failure.
   */
  public function updateDeal(string $deal_id, array $properties, array $associations = []): ?array {
    try {
      $client = $this->getClient();
      if (!$client) {
        return NULL;
      }

      $dealInput = new \HubSpot\Client\Crm\Deals\Model\SimplePublicObjectInput([
        'properties' => $properties,
      ]);

      $response = $client->crm()->deals()->basicApi()->update($deal_id, $dealInput);

      // Add associations if provided.
      if (!empty($associations)) {
        $this->createAssociations(self::OBJECT_TYPE_DEAL, $deal_id, $associations);
      }

      return $this->formatObjectResponse($response);
    }
    catch (\Exception $e) {
      $this->handleApiException($e, 'update deal');
      return NULL;
    }
  }

  /**
   * Gets a deal from HubSpot.
   *
   * @param string $deal_id
   *   The deal ID.
   * @param array $properties
   *   The properties to retrieve.
   * @param bool $include_associations
   *   Whether to include associations.
   *
   * @return array|null
   *   The deal data or NULL on failure.
   */
  public function getDeal(string $deal_id, array $properties = [], bool $include_associations = FALSE): ?array {
    try {
      $client = $this->getClient();
      if (!$client) {
        return NULL;
      }

      $associations = $include_associations ? ['contacts', 'companies'] : NULL;

      $response = $client->crm()->deals()->basicApi()->getById(
        $deal_id,
        $properties,
        NULL,
        $associations,
        FALSE
      );

      return $this->formatObjectResponse($response);
    }
    catch (\Exception $e) {
      $this->handleApiException($e, 'get deal');
      return NULL;
    }
  }

  /**
   * Deletes a deal from HubSpot.
   *
   * @param string $deal_id
   *   The deal ID.
   *
   * @return bool
   *   TRUE on success, FALSE on failure.
   */
  public function deleteDeal(string $deal_id): bool {
    try {
      $client = $this->getClient();
      if (!$client) {
        return FALSE;
      }

      $client->crm()->deals()->basicApi()->archive($deal_id);

      return TRUE;
    }
    catch (\Exception $e) {
      $this->handleApiException($e, 'delete deal');
      return FALSE;
    }
  }

  /**
   * Searches for deals in HubSpot.
   *
   * @param array $filters
   *   The search filters.
   * @param string $sort_by
   *   The property to sort by.
   * @param int $limit
   *   The maximum number of results.
   *
   * @return array|null
   *   The search results or NULL on failure.
   */
  public function searchDeals(array $filters, string $sort_by = '', int $limit = 100): ?array {
    return $this->performSearch('deals', $filters, $sort_by, $limit);
  }

  /**
   * Creates a lead in HubSpot.
   *
   * @param array $properties
   *   The lead properties.
   * @param array $associations
   *   Optional associations to other objects (required: contact).
   *
   * @return array|null
   *   The created lead data or NULL on failure.
   */
  public function createLead(array $properties, array $associations = []): ?array {
    try {
      $client = $this->getClient();
      if (!$client) {
        return NULL;
      }

      $leadInput = new \HubSpot\Client\Crm\Objects\Model\SimplePublicObjectInputForCreate([
        'properties' => $properties,
        'associations' => $this->formatAssociations($associations, self::OBJECT_TYPE_LEAD),
      ]);

      $response = $client->crm()->objects()->basicApi()->create('leads', $leadInput);

      return $this->formatObjectResponse($response);
    }
    catch (\Exception $e) {
      $this->handleApiException($e, 'create lead');
      return NULL;
    }
  }

  /**
   * Updates a lead in HubSpot.
   *
   * @param string $lead_id
   *   The lead ID.
   * @param array $properties
   *   The properties to update.
   * @param array $associations
   *   Optional associations to add to the lead.
   *
   * @return array|null
   *   The updated lead data or NULL on failure.
   */
  public function updateLead(string $lead_id, array $properties, array $associations = []): ?array {
    try {
      $client = $this->getClient();
      if (!$client) {
        return NULL;
      }

      $leadInput = new \HubSpot\Client\Crm\Objects\Model\SimplePublicObjectInput([
        'properties' => $properties,
      ]);

      $response = $client->crm()->objects()->basicApi()->update('leads', $lead_id, $leadInput);

      // Add associations if provided.
      if (!empty($associations)) {
        $this->createAssociations(self::OBJECT_TYPE_LEAD, $lead_id, $associations);
      }

      return $this->formatObjectResponse($response);
    }
    catch (\Exception $e) {
      $this->handleApiException($e, 'update lead');
      return NULL;
    }
  }

  /**
   * Gets a lead from HubSpot.
   *
   * @param string $lead_id
   *   The lead ID.
   * @param array $properties
   *   The properties to retrieve.
   * @param bool $include_associations
   *   Whether to include associations.
   *
   * @return array|null
   *   The lead data or NULL on failure.
   */
  public function getLead(string $lead_id, array $properties = [], bool $include_associations = FALSE): ?array {
    try {
      $client = $this->getClient();
      if (!$client) {
        return NULL;
      }

      $associations = $include_associations ? ['contacts', 'companies'] : NULL;

      $response = $client->crm()->objects()->basicApi()->getById(
        'leads',
        $lead_id,
        $properties,
        NULL,
        $associations,
        FALSE
      );

      return $this->formatObjectResponse($response);
    }
    catch (\Exception $e) {
      $this->handleApiException($e, 'get lead');
      return NULL;
    }
  }

  /**
   * Deletes a lead from HubSpot.
   *
   * @param string $lead_id
   *   The lead ID.
   *
   * @return bool
   *   TRUE on success, FALSE on failure.
   */
  public function deleteLead(string $lead_id): bool {
    try {
      $client = $this->getClient();
      if (!$client) {
        return FALSE;
      }

      $client->crm()->objects()->basicApi()->archive('leads', $lead_id);

      return TRUE;
    }
    catch (\Exception $e) {
      $this->handleApiException($e, 'delete lead');
      return FALSE;
    }
  }

  /**
   * Searches for leads in HubSpot.
   *
   * @param array $filters
   *   The search filters.
   * @param string $sort_by
   *   The property to sort by.
   * @param int $limit
   *   The maximum number of results.
   *
   * @return array|null
   *   The search results or NULL on failure.
   */
  public function searchLeads(array $filters, string $sort_by = '', int $limit = 100): ?array {
    return $this->performSearch('leads', $filters, $sort_by, $limit);
  }

  /**
   * Creates a ticket in HubSpot.
   *
   * @param array $properties
   *   The ticket properties.
   * @param array $associations
   *   Optional associations to other objects.
   *
   * @return array|null
   *   The created ticket data or NULL on failure.
   */
  public function createTicket(array $properties, array $associations = []): ?array {
    try {
      $client = $this->getClient();
      if (!$client) {
        return NULL;
      }

      $ticketInput = new \HubSpot\Client\Crm\Tickets\Model\SimplePublicObjectInputForCreate([
        'properties' => $properties,
        'associations' => $this->formatAssociations($associations, self::OBJECT_TYPE_TICKET),
      ]);

      $response = $client->crm()->tickets()->basicApi()->create($ticketInput);

      return $this->formatObjectResponse($response);
    }
    catch (\Exception $e) {
      $this->handleApiException($e, 'create ticket');
      return NULL;
    }
  }

  /**
   * Updates a ticket in HubSpot.
   *
   * @param string $ticket_id
   *   The ticket ID.
   * @param array $properties
   *   The properties to update.
   * @param array $associations
   *   Optional associations to add to the ticket.
   *
   * @return array|null
   *   The updated ticket data or NULL on failure.
   */
  public function updateTicket(string $ticket_id, array $properties, array $associations = []): ?array {
    try {
      $client = $this->getClient();
      if (!$client) {
        return NULL;
      }

      $ticketInput = new \HubSpot\Client\Crm\Tickets\Model\SimplePublicObjectInput([
        'properties' => $properties,
      ]);

      $response = $client->crm()->tickets()->basicApi()->update($ticket_id, $ticketInput);

      // Add associations if provided.
      if (!empty($associations)) {
        $this->createAssociations(self::OBJECT_TYPE_TICKET, $ticket_id, $associations);
      }

      return $this->formatObjectResponse($response);
    }
    catch (\Exception $e) {
      $this->handleApiException($e, 'update ticket');
      return NULL;
    }
  }

  /**
   * Gets a ticket from HubSpot.
   *
   * @param string $ticket_id
   *   The ticket ID.
   * @param array $properties
   *   The properties to retrieve.
   * @param bool $include_associations
   *   Whether to include associations.
   *
   * @return array|null
   *   The ticket data or NULL on failure.
   */
  public function getTicket(string $ticket_id, array $properties = [], bool $include_associations = FALSE): ?array {
    try {
      $client = $this->getClient();
      if (!$client) {
        return NULL;
      }

      $associations = $include_associations ? ['contacts', 'companies', 'deals'] : NULL;

      $response = $client->crm()->tickets()->basicApi()->getById(
        $ticket_id,
        $properties,
        NULL,
        $associations,
        FALSE
      );

      return $this->formatObjectResponse($response);
    }
    catch (\Exception $e) {
      $this->handleApiException($e, 'get ticket');
      return NULL;
    }
  }

  /**
   * Deletes a ticket from HubSpot.
   *
   * @param string $ticket_id
   *   The ticket ID.
   *
   * @return bool
   *   TRUE on success, FALSE on failure.
   */
  public function deleteTicket(string $ticket_id): bool {
    try {
      $client = $this->getClient();
      if (!$client) {
        return FALSE;
      }

      $client->crm()->tickets()->basicApi()->archive($ticket_id);

      return TRUE;
    }
    catch (\Exception $e) {
      $this->handleApiException($e, 'delete ticket');
      return FALSE;
    }
  }

  /**
   * Searches for tickets in HubSpot.
   *
   * @param array $filters
   *   The search filters.
   * @param string $sort_by
   *   The property to sort by.
   * @param int $limit
   *   The maximum number of results.
   *
   * @return array|null
   *   The search results or NULL on failure.
   */
  public function searchTickets(array $filters, string $sort_by = '', int $limit = 100): ?array {
    return $this->performSearch('tickets', $filters, $sort_by, $limit);
  }

  /**
   * Associates two objects in HubSpot.
   *
   * @param string $from_object_type
   *   The source object type.
   * @param string $from_object_id
   *   The source object ID.
   * @param string $to_object_type
   *   The target object type.
   * @param string $to_object_id
   *   The target object ID.
   *
   * @return bool
   *   TRUE on success, FALSE on failure.
   */
  public function associateObjects(string $from_object_type, string $from_object_id, string $to_object_type, string $to_object_id): bool {
    try {
      $client = $this->getClient();
      if (!$client) {
        return FALSE;
      }

      // Get the association type ID.
      $associationTypeId = $this->getAssociationTypeId($from_object_type, $to_object_type);

      // Use v4 associations API.
      $client->crm()->associations()->v4()->basicApi()->create(
        $from_object_type,
        $from_object_id,
        $to_object_type,
        $to_object_id,
        [
          [
            'associationCategory' => 'HUBSPOT_DEFINED',
            'associationTypeId' => $associationTypeId,
          ],
        ]
      );

      return TRUE;
    }
    catch (\Exception $e) {
      $this->handleApiException($e, 'associate objects');
      return FALSE;
    }
  }

  /**
   * Disassociates two objects in HubSpot.
   *
   * @param string $from_object_type
   *   The source object type.
   * @param string $from_object_id
   *   The source object ID.
   * @param string $to_object_type
   *   The target object type.
   * @param string $to_object_id
   *   The target object ID.
   *
   * @return bool
   *   TRUE on success, FALSE on failure.
   */
  public function disassociateObjects(string $from_object_type, string $from_object_id, string $to_object_type, string $to_object_id): bool {
    try {
      $client = $this->getClient();
      if (!$client) {
        return FALSE;
      }

      // Get the association type ID.
      $associationTypeId = $this->getAssociationTypeId($from_object_type, $to_object_type);

      // Use v4 associations API.
      $client->crm()->associations()->v4()->basicApi()->archive(
        $from_object_type,
        $from_object_id,
        $to_object_type,
        $to_object_id,
        [
          [
            'associationCategory' => 'HUBSPOT_DEFINED',
            'associationTypeId' => $associationTypeId,
          ],
        ]
      );

      return TRUE;
    }
    catch (\Exception $e) {
      $this->handleApiException($e, 'disassociate objects');
      return FALSE;
    }
  }

  /**
   * Gets associations for an object in HubSpot.
   *
   * @param string $object_type
   *   The object type.
   * @param string $object_id
   *   The object ID.
   * @param string $to_object_type
   *   The type of objects to get associations for.
   *
   * @return array|null
   *   The associations data or NULL on failure.
   */
  public function getAssociations(string $object_type, string $object_id, string $to_object_type): ?array {
    try {
      $client = $this->getClient();
      if (!$client) {
        return NULL;
      }

      // Use v4 associations API which works across all object types.
      $response = $client->crm()->associations()->v4()->basicApi()->getPage(
        $object_type,
        $object_id,
        $to_object_type
      );

      // Convert response to array of IDs.
      $associations = [];
      if ($response && method_exists($response, 'getResults')) {
        foreach ($response->getResults() as $result) {
          if (method_exists($result, 'getToObjectId')) {
            $associations[] = $result->getToObjectId();
          }
        }
      }

      return $associations;
    }
    catch (\Exception $e) {
      $this->handleApiException($e, 'get associations');
      return NULL;
    }
  }

  /**
   * Creates a note in HubSpot.
   *
   * @param array $properties
   *   The note properties (including hs_note_body).
   * @param array $associations
   *   The objects to associate with the note.
   *
   * @return array|null
   *   The created note data or NULL on failure.
   */
  public function createNote(array $properties, array $associations = []): ?array {
    try {
      $client = $this->getClient();
      if (!$client) {
        return NULL;
      }

      // Add timestamp if not provided.
      if (!isset($properties['hs_timestamp'])) {
        $properties['hs_timestamp'] = (new \DateTime())->format('c');
      }

      $noteInput = new \HubSpot\Client\Crm\Objects\Notes\Model\SimplePublicObjectInputForCreate([
        'properties' => $properties,
        'associations' => $this->formatAssociations($associations, self::OBJECT_TYPE_NOTE),
      ]);

      $response = $client->crm()->objects()->notes()->basicApi()->create($noteInput);

      return $this->formatObjectResponse($response);
    }
    catch (\Exception $e) {
      $this->handleApiException($e, 'create note');
      return NULL;
    }
  }

  /**
   * Creates a task in HubSpot.
   *
   * @param array $task_data
   *   The task data including title, body, due date, priority, type.
   * @param array $associations
   *   The objects to associate with the task.
   *
   * @return array|null
   *   The created task data or NULL on failure.
   */
  public function createTask(array $task_data, array $associations): ?array {
    try {
      $client = $this->getClient();
      if (!$client) {
        return NULL;
      }

      // Create task without associations first.
      $taskInput = new \HubSpot\Client\Crm\Objects\Tasks\Model\SimplePublicObjectInputForCreate([
        'properties' => $task_data,
      ]);

      $response = $client->crm()->objects()->tasks()->basicApi()->create($taskInput);
      $task_id = $response->getId();

      // Create associations separately if provided.
      if (!empty($associations)) {
        $this->createAssociations(self::OBJECT_TYPE_TASK, $task_id, $associations);
      }

      return $this->formatObjectResponse($response);
    }
    catch (\Exception $e) {
      $this->handleApiException($e, 'create task');
      return NULL;
    }
  }

  /**
   * Updates a note in HubSpot.
   *
   * @param string $note_id
   *   The note ID.
   * @param array $properties
   *   The note properties to update (including hs_note_body).
   * @param array $associations
   *   Optional associations to other objects.
   *
   * @return array|null
   *   The updated note data or NULL on failure.
   */
  public function updateNote(string $note_id, array $properties, array $associations = []): ?array {
    try {
      $client = $this->getClient();
      if (!$client) {
        return NULL;
      }

      $noteInput = new \HubSpot\Client\Crm\Objects\Notes\Model\SimplePublicObjectInput([
        'properties' => $properties,
      ]);

      $response = $client->crm()->objects()->notes()->basicApi()->update($note_id, $noteInput);

      // Create associations if provided.
      if (!empty($associations)) {
        $this->createAssociations(self::OBJECT_TYPE_NOTE, $note_id, $associations);
      }

      return $this->formatObjectResponse($response);
    }
    catch (\Exception $e) {
      $this->handleApiException($e, 'update note');
      return NULL;
    }
  }

  /**
   * Gets a note from HubSpot.
   *
   * @param string $note_id
   *   The note ID.
   * @param array $properties
   *   The properties to retrieve.
   *
   * @return array|null
   *   The note data or NULL on failure.
   */
  public function getNote(string $note_id, array $properties = []): ?array {
    try {
      $client = $this->getClient();
      if (!$client) {
        return NULL;
      }

      $response = $client->crm()->objects()->notes()->basicApi()->getById(
        $note_id,
        $properties,
        NULL,
        NULL,
        FALSE
      );

      return $this->formatObjectResponse($response);
    }
    catch (\Exception $e) {
      $this->handleApiException($e, 'get note');
      return NULL;
    }
  }

  /**
   * Deletes a note from HubSpot.
   *
   * @param string $note_id
   *   The note ID.
   *
   * @return bool
   *   TRUE on success, FALSE on failure.
   */
  public function deleteNote(string $note_id): bool {
    try {
      $client = $this->getClient();
      if (!$client) {
        return FALSE;
      }

      $client->crm()->objects()->notes()->basicApi()->archive($note_id);

      return TRUE;
    }
    catch (\Exception $e) {
      $this->handleApiException($e, 'delete note');
      return FALSE;
    }
  }

  /**
   * Searches for notes in HubSpot.
   *
   * @param array $filters
   *   The search filters.
   * @param string $sort_by
   *   The property to sort by.
   * @param int $limit
   *   The maximum number of results.
   *
   * @return array|null
   *   The search results or NULL on failure.
   */
  public function searchNotes(array $filters, string $sort_by = '', int $limit = 100): ?array {
    return $this->performSearch('notes', $filters, $sort_by, $limit);
  }

  /**
   * Updates a task in HubSpot.
   *
   * @param string $task_id
   *   The task ID.
   * @param array $task_data
   *   The task properties to update.
   * @param array $associations
   *   Optional associations to other objects.
   *
   * @return array|null
   *   The updated task data or NULL on failure.
   */
  public function updateTask(string $task_id, array $task_data, array $associations = []): ?array {
    try {
      $client = $this->getClient();
      if (!$client) {
        return NULL;
      }

      $taskInput = new \HubSpot\Client\Crm\Objects\Tasks\Model\SimplePublicObjectInput([
        'properties' => $task_data,
      ]);

      $response = $client->crm()->objects()->tasks()->basicApi()->update($task_id, $taskInput);

      // Create associations if provided.
      if (!empty($associations)) {
        $this->createAssociations(self::OBJECT_TYPE_TASK, $task_id, $associations);
      }

      return $this->formatObjectResponse($response);
    }
    catch (\Exception $e) {
      $this->handleApiException($e, 'update task');
      return NULL;
    }
  }

  /**
   * Gets a task from HubSpot.
   *
   * @param string $task_id
   *   The task ID.
   * @param array $properties
   *   The properties to retrieve.
   *
   * @return array|null
   *   The task data or NULL on failure.
   */
  public function getTask(string $task_id, array $properties = []): ?array {
    try {
      $client = $this->getClient();
      if (!$client) {
        return NULL;
      }

      $response = $client->crm()->objects()->tasks()->basicApi()->getById(
        $task_id,
        $properties,
        NULL,
        NULL,
        FALSE
      );

      return $this->formatObjectResponse($response);
    }
    catch (\Exception $e) {
      $this->handleApiException($e, 'get task');
      return NULL;
    }
  }

  /**
   * Deletes a task from HubSpot.
   *
   * @param string $task_id
   *   The task ID.
   *
   * @return bool
   *   TRUE on success, FALSE on failure.
   */
  public function deleteTask(string $task_id): bool {
    try {
      $client = $this->getClient();
      if (!$client) {
        return FALSE;
      }

      $client->crm()->objects()->tasks()->basicApi()->archive($task_id);

      return TRUE;
    }
    catch (\Exception $e) {
      $this->handleApiException($e, 'delete task');
      return FALSE;
    }
  }

  /**
   * Searches for tasks in HubSpot.
   *
   * @param array $filters
   *   The search filters.
   * @param string $sort_by
   *   The property to sort by.
   * @param int $limit
   *   The maximum number of results.
   *
   * @return array|null
   *   The search results or NULL on failure.
   */
  public function searchTasks(array $filters, string $sort_by = '', int $limit = 100): ?array {
    return $this->performSearch('tasks', $filters, $sort_by, $limit);
  }

  /**
   * Lists all pipelines for an object type.
   *
   * @param string $object_type
   *   The object type (deals, tickets, leads, etc.).
   *
   * @return array|null
   *   Array of pipelines with stages or NULL on failure.
   */
  public function listPipelines(string $object_type): ?array {
    try {
      $client = $this->getClient();
      if (!$client) {
        return NULL;
      }

      $response = $client->crm()->pipelines()->pipelinesApi()->getAll($object_type);

      $pipelines = [];
      if ($response && method_exists($response, 'getResults')) {
        foreach ($response->getResults() as $pipeline) {
          $pipelineData = [
            'id' => $pipeline->getId(),
            'label' => $pipeline->getLabel(),
            'displayOrder' => $pipeline->getDisplayOrder(),
            'stages' => [],
          ];

          if (method_exists($pipeline, 'getStages')) {
            foreach ($pipeline->getStages() as $stage) {
              $pipelineData['stages'][] = [
                'id' => $stage->getId(),
                'label' => $stage->getLabel(),
                'displayOrder' => $stage->getDisplayOrder(),
                'metadata' => $stage->getMetadata(),
              ];
            }
          }

          $pipelines[] = $pipelineData;
        }
      }

      return $pipelines;
    }
    catch (\Exception $e) {
      $this->handleApiException($e, 'list pipelines');
      return NULL;
    }
  }

  /**
   * Gets a specific pipeline by ID.
   *
   * @param string $object_type
   *   The object type (deals, tickets, leads, etc.).
   * @param string $pipeline_id
   *   The pipeline ID.
   *
   * @return array|null
   *   Pipeline data with stages or NULL on failure.
   */
  public function getPipeline(string $object_type, string $pipeline_id): ?array {
    try {
      $client = $this->getClient();
      if (!$client) {
        return NULL;
      }

      $response = $client->crm()->pipelines()->pipelinesApi()->getById($object_type, $pipeline_id);

      $pipelineData = [
        'id' => $response->getId(),
        'label' => $response->getLabel(),
        'displayOrder' => $response->getDisplayOrder(),
        'stages' => [],
      ];

      if (method_exists($response, 'getStages')) {
        foreach ($response->getStages() as $stage) {
          $pipelineData['stages'][] = [
            'id' => $stage->getId(),
            'label' => $stage->getLabel(),
            'displayOrder' => $stage->getDisplayOrder(),
            'metadata' => $stage->getMetadata(),
          ];
        }
      }

      return $pipelineData;
    }
    catch (\Exception $e) {
      $this->handleApiException($e, 'get pipeline');
      return NULL;
    }
  }

  /**
   * Performs a search for any object type in HubSpot.
   *
   * @param string $object_type
   *   The object type to search (contacts, companies, deals, leads, tickets, notes, tasks).
   * @param array $filters
   *   The search filters.
   * @param string $sort_by
   *   The property to sort by.
   * @param int $limit
   *   The maximum number of results.
   *
   * @return array|null
   *   The search results or NULL on failure.
   */
  protected function performSearch(string $object_type, array $filters, string $sort_by = '', int $limit = 100): ?array {
    try {
      $client = $this->getClient();
      if (!$client) {
        return NULL;
      }

      // Determine namespace and API path based on object type.
      $namespace_map = [
        'contacts' => 'Contacts',
        'companies' => 'Companies',
        'deals' => 'Deals',
        'tickets' => 'Tickets',
        'leads' => 'Objects',
        'notes' => 'Objects',
        'tasks' => 'Objects',
      ];

      $namespace = $namespace_map[$object_type] ?? 'Objects';
      $filter_class = "\\HubSpot\\Client\\Crm\\{$namespace}\\Model\\Filter";
      $filter_group_class = "\\HubSpot\\Client\\Crm\\{$namespace}\\Model\\FilterGroup";
      $search_request_class = "\\HubSpot\\Client\\Crm\\{$namespace}\\Model\\PublicObjectSearchRequest";

      // Build filter groups.
      $filterGroups = [];
      if (!empty($filters)) {
        $filterObjects = [];
        foreach ($filters as $filterData) {
          $filterObject = new $filter_class();
          $filterObject->setPropertyName($filterData['propertyName']);
          $filterObject->setOperator($filterData['operator']);

          // Set value or values depending on what's provided.
          if (isset($filterData['values'])) {
            $filterObject->setValues($filterData['values']);
          }
          elseif (isset($filterData['value'])) {
            $filterObject->setValue($filterData['value']);
          }

          // Set highValue for range queries.
          if (isset($filterData['highValue'])) {
            $filterObject->setHighValue($filterData['highValue']);
          }

          $filterObjects[] = $filterObject;
        }

        $filterGroup = new $filter_group_class();
        $filterGroup->setFilters($filterObjects);
        $filterGroups[] = $filterGroup;
      }

      // Build search request.
      $searchRequest = new $search_request_class();
      $searchRequest->setFilterGroups($filterGroups);
      $searchRequest->setLimit($limit);

      if (!empty($sort_by)) {
        $searchRequest->setSorts([
          [
            'propertyName' => $sort_by,
            'direction' => 'ASCENDING',
          ],
        ]);
      }

      // Execute search based on object type.
      if (in_array($object_type, ['leads', 'notes', 'tasks'])) {
        $response = $client->crm()->objects()->searchApi()->doSearch($object_type, $searchRequest);
      }
      else {
        $api_method = $object_type;
        $response = $client->crm()->$api_method()->searchApi()->doSearch($searchRequest);
      }

      return [
        'total' => $response->getTotal(),
        'results' => array_map([$this, 'formatObjectResponse'], $response->getResults()),
      ];
    }
    catch (\Exception $e) {
      $this->handleApiException($e, "search {$object_type}");
      return NULL;
    }
  }

  /**
   * Formats a HubSpot API response object to an array.
   *
   * @param mixed $response
   *   The API response object.
   *
   * @return array
   *   The formatted array with only scalar values.
   */
  protected function formatObjectResponse(mixed $response): array {
    if (is_object($response)) {
      $data = [
        'id' => $response->getId(),
        'properties' => [],
        'created_at' => '',
        'updated_at' => '',
      ];

      // Convert properties to scalar values.
      $properties = $response->getProperties();
      if (is_array($properties)) {
        $data['properties'] = $properties;
      }
      elseif (is_object($properties)) {
        $data['properties'] = json_decode(json_encode($properties), TRUE);
      }

      // Convert DateTime objects to strings.
      $created_at = $response->getCreatedAt();
      if ($created_at instanceof \DateTime) {
        $data['created_at'] = $created_at->format('c');
      }
      elseif (is_string($created_at)) {
        $data['created_at'] = $created_at;
      }

      $updated_at = $response->getUpdatedAt();
      if ($updated_at instanceof \DateTime) {
        $data['updated_at'] = $updated_at->format('c');
      }
      elseif (is_string($updated_at)) {
        $data['updated_at'] = $updated_at;
      }

      // Handle associations if present.
      if (method_exists($response, 'getAssociations')) {
        $associations = $response->getAssociations();
        if (is_array($associations)) {
          $data['associations'] = $associations;
        }
        elseif (is_object($associations)) {
          $data['associations'] = json_decode(json_encode($associations), TRUE);
        }
      }

      return $data;
    }

    return [];
  }

  /**
   * Formats associations array for HubSpot API.
   *
   * @param array $associations
   *   The associations array with object type as key and IDs as values.
   * @param string $from_object_type
   *   The type of object being created (e.g., 'deal').
   *
   * @return array
   *   The formatted associations.
   */
  protected function formatAssociations(array $associations, string $from_object_type = 'deal'): array {
    $formatted = [];

    foreach ($associations as $to_object_type => $ids) {
      if (!is_array($ids)) {
        $ids = [$ids];
      }

      foreach ($ids as $id) {
        $formatted[] = [
          'to' => [
            'id' => (string) $id,
          ],
          'types' => [
            [
              'associationCategory' => 'HUBSPOT_DEFINED',
              'associationTypeId' => $this->getAssociationTypeId($from_object_type, $to_object_type),
            ],
          ],
        ];
      }
    }

    return $formatted;
  }

  /**
   * Creates associations between objects in HubSpot.
   *
   * @param string $from_object_type
   *   The source object type (e.g., 'deal', 'contact').
   * @param string $from_object_id
   *   The source object ID.
   * @param array $associations
   *   Array of associations keyed by object type with array of IDs as values.
   *
   * @return bool
   *   TRUE on success, FALSE on failure.
   */
  protected function createAssociations(string $from_object_type, string $from_object_id, array $associations): bool {
    try {
      $client = $this->getClient();
      if (!$client) {
        return FALSE;
      }

      foreach ($associations as $to_object_type => $ids) {
        if (!is_array($ids)) {
          $ids = [$ids];
        }

        foreach ($ids as $to_object_id) {
          $associationTypeId = $this->getAssociationTypeId($from_object_type, $to_object_type);

          $this->logger->debug('Creating association: from=@from(@from_id) to=@to(@to_id) typeId=@type', [
            '@from' => $from_object_type,
            '@from_id' => $from_object_id,
            '@to' => $to_object_type,
            '@to_id' => $to_object_id,
            '@type' => $associationTypeId,
          ]);

          // Use the associations API to create the association.
          $client->crm()->associations()->v4()->basicApi()->create(
            $from_object_type,
            $from_object_id,
            $to_object_type,
            $to_object_id,
            [
              [
                'associationCategory' => 'HUBSPOT_DEFINED',
                'associationTypeId' => $associationTypeId,
              ],
            ]
          );
        }
      }

      return TRUE;
    }
    catch (\Exception $e) {
      $this->handleApiException($e, 'create associations');
      return FALSE;
    }
  }

  /**
   * Gets the association type ID for a relationship between two object types.
   *
   * @param string $from_object_type
   *   The source object type.
   * @param string $to_object_type
   *   The target object type.
   *
   * @return int
   *   The association type ID.
   */
  protected function getAssociationTypeId(string $from_object_type, string $to_object_type): int {
    // Association type IDs for common relationships.
    // Format: from_type|to_type => type_id.
    $types = [
      // Deal associations.
      'deal|contact' => 3,
      'deal|company' => 5,
      'deal|ticket' => 27,
      'deal|note' => 213,
      'deal|task' => 215,
      // Contact associations.
      'contact|company' => 1,
      'contact|deal' => 4,
      'contact|ticket' => 15,
      'contact|note' => 201,
      'contact|task' => 203,
      // Company associations.
      'company|contact' => 2,
      'company|deal' => 6,
      'company|ticket' => 25,
      'company|note' => 189,
      'company|task' => 191,
      // Ticket associations.
      'ticket|contact' => 16,
      'ticket|company' => 26,
      'ticket|deal' => 28,
      'ticket|note' => 227,
      'ticket|task' => 229,
      // Note associations.
      'note|contact' => 202,
      'note|company' => 190,
      'note|deal' => 214,
      'note|ticket' => 228,
      // Task associations.
      'task|contact' => 204,
      'task|company' => 192,
      'task|deal' => 216,
      'task|ticket' => 230,
    ];

    $key = $from_object_type . '|' . $to_object_type;
    return $types[$key] ?? 1;
  }

  /**
   * Gets the association type label.
   *
   * @param string $from_type
   *   The source object type.
   * @param string $to_type
   *   The target object type.
   *
   * @return string
   *   The association type label.
   */
  protected function getAssociationType(string $from_type, string $to_type): string {
    // HubSpot uses specific association type strings.
    $types = [
      self::OBJECT_TYPE_CONTACT => [
        self::OBJECT_TYPE_COMPANY => 'contact_to_company',
        self::OBJECT_TYPE_DEAL => 'contact_to_deal',
        self::OBJECT_TYPE_TICKET => 'contact_to_ticket',
      ],
      self::OBJECT_TYPE_COMPANY => [
        self::OBJECT_TYPE_CONTACT => 'company_to_contact',
        self::OBJECT_TYPE_DEAL => 'company_to_deal',
        self::OBJECT_TYPE_TICKET => 'company_to_ticket',
      ],
      self::OBJECT_TYPE_DEAL => [
        self::OBJECT_TYPE_CONTACT => 'deal_to_contact',
        self::OBJECT_TYPE_COMPANY => 'deal_to_company',
        self::OBJECT_TYPE_TICKET => 'deal_to_ticket',
      ],
      self::OBJECT_TYPE_TICKET => [
        self::OBJECT_TYPE_CONTACT => 'ticket_to_contact',
        self::OBJECT_TYPE_COMPANY => 'ticket_to_company',
        self::OBJECT_TYPE_DEAL => 'ticket_to_deal',
      ],
    ];

    return $types[$from_type][$to_type] ?? 'contact_to_company';
  }

  /**
   * Handles API exceptions with consistent logging.
   *
   * @param \Exception $e
   *   The exception.
   * @param string $operation
   *   The operation that failed.
   */
  protected function handleApiException(\Exception $e, string $operation): void {
    $this->logger->error('Failed to @operation: @message', [
      '@operation' => $operation,
      '@message' => $e->getMessage(),
    ]);
  }

}
