<?php

declare(strict_types=1);

namespace Drupal\eca_hubspot;

/**
 * Trait for consistent datetime formatting for HubSpot API services.
 */
trait DateTimeFormattingTrait {

  /**
   * Formats datetime value for HubSpot API (Unix timestamp in milliseconds).
   *
   * @param mixed $datetime
   *   The datetime value to format (string, DateTime, int, etc).
   * @param string|null $timezone
   *   Optional timezone identifier.
   *
   * @return int|null
   *   Unix timestamp in milliseconds or NULL on failure.
   */
  public function formatDateTimeForApi(mixed $datetime, ?string $timezone = null): ?int {
    if (empty($datetime)) {
      return NULL;
    }

    try {
      $dt = NULL;

      // Handle different input types
      if ($datetime instanceof \DateTime) {
        // Already a DateTime object (e.g., from Drupal field)
        $dt = clone $datetime;
      }
      elseif ($datetime instanceof \DateTimeInterface) {
        // Other DateTime interface implementations
        $dt = \DateTime::createFromInterface($datetime);
      }
      elseif (is_numeric($datetime)) {
        // Unix timestamp - check if it's already in milliseconds
        if ($datetime > 9999999999) {
          // Already in milliseconds, return as-is
          return (int) $datetime;
        }
        // Convert from seconds to milliseconds
        return (int) ($datetime * 1000);
      }
      elseif (is_string($datetime)) {
        // String datetime - try various formats
        $dt = $this->parseStringDateTime($datetime);
      }
      elseif (is_array($datetime) && isset($datetime['date'])) {
        // Drupal field array format
        $dt = new \DateTime($datetime['date']);
      }

      if (!$dt) {
        if (method_exists($this, 'logger') && isset($this->logger)) {
          $this->logger->error('Unable to parse datetime value: @value', [
            '@value' => is_scalar($datetime) ? $datetime : gettype($datetime),
          ]);
        }
        return NULL;
      }

      // Apply timezone if provided, otherwise use Drupal default
      $target_timezone = NULL;
      if (!empty($timezone)) {
        $target_timezone = $timezone;
      }
      else {
        // Use Drupal's default timezone if no timezone specified
        $target_timezone = \Drupal::config('system.date')->get('timezone.default') ?? date_default_timezone_get();
      }

      if (!empty($target_timezone)) {
        try {
          $dt->setTimezone(new \DateTimeZone($target_timezone));
        }
        catch (\Exception $e) {
          if (method_exists($this, 'logger') && isset($this->logger)) {
            $this->logger->warning('Invalid timezone "@tz", using datetime as-is: @message', [
              '@tz' => $target_timezone,
              '@message' => $e->getMessage(),
            ]);
          }
        }
      }

      // Return Unix timestamp in milliseconds for HubSpot API
      return (int) ($dt->getTimestamp() * 1000);
    }
    catch (\Exception $e) {
      if (method_exists($this, 'logger') && isset($this->logger)) {
        $this->logger->error('Error formatting datetime for HubSpot API: @message', [
          '@message' => $e->getMessage(),
        ]);
      }
      return NULL;
    }
  }

  /**
   * Parses string datetime with fallback formats.
   *
   * @param string $datetime
   *   The datetime string to parse.
   *
   * @return \DateTime|null
   *   Parsed DateTime object or NULL on failure.
   */
  private function parseStringDateTime(string $datetime): ?\DateTime {
    // Try general parsing first (handles most standard formats automatically)
    try {
      return new \DateTime($datetime);
    }
    catch (\Exception $e) {
      // General parsing failed, try specific formats for edge cases
    }

    // Specific formats for cases that general parsing can't handle
    $formats = [
      'D, d M Y - H:i',        // Fri, 12 Sep 2025 - 10:00 (Drupal default format)
      'd M Y - H:i',           // 12 Sep 2025 - 10:00 (without day name)
    ];

    // Try exact format matches for special cases
    foreach ($formats as $format) {
      $dt = \DateTime::createFromFormat($format, $datetime);
      if ($dt !== FALSE) {
        return $dt;
      }
    }

    return NULL;
  }

}
