<?php

declare(strict_types=1);

namespace Drupal\eca_hubspot\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;
use Drupal\eca_hubspot\Service\HubSpotService;

/**
 * Create a HubSpot contact.
 */
#[Action(
  id: 'eca_hubspot_create_contact',
  label: new TranslatableMarkup('HubSpot: Create Contact'),
  category: new TranslatableMarkup('HubSpot'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Create a new contact record in HubSpot.'),
  version_introduced: '1.0.0',
)]
final class CreateContact extends HubSpotActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'firstname' => '',
      'lastname' => '',
      'email' => '',
      'phone' => '',
      'company' => '',
      'lifecyclestage' => '',
      'lifecyclestage_token' => '',
      'additional_properties' => '',
      'token_name' => '',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['email'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Email'),
      '#description' => $this->t('The contact email address (optional).'),
      '#default_value' => $this->configuration['email'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['firstname'] = [
      '#type' => 'textfield',
      '#title' => $this->t('First Name'),
      '#description' => $this->t('The contact first name.'),
      '#default_value' => $this->configuration['firstname'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['lastname'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Last Name'),
      '#description' => $this->t('The contact last name.'),
      '#default_value' => $this->configuration['lastname'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['phone'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Phone'),
      '#description' => $this->t('The contact phone number.'),
      '#default_value' => $this->configuration['phone'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['company'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Company Name'),
      '#description' => $this->t('The company name associated with this contact.'),
      '#default_value' => $this->configuration['company'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['lifecyclestage'] = [
      '#type' => 'select',
      '#title' => $this->t('Lifecycle Stage'),
      '#description' => $this->t('The lifecycle stage of the contact.'),
      '#options' => [
        '' => $this->t('- None -'),
        HubSpotService::LIFECYCLE_SUBSCRIBER => $this->t('Subscriber'),
        HubSpotService::LIFECYCLE_LEAD => $this->t('Lead'),
        HubSpotService::LIFECYCLE_MQL => $this->t('Marketing Qualified Lead'),
        HubSpotService::LIFECYCLE_SQL => $this->t('Sales Qualified Lead'),
        HubSpotService::LIFECYCLE_OPPORTUNITY => $this->t('Opportunity'),
        HubSpotService::LIFECYCLE_CUSTOMER => $this->t('Customer'),
        HubSpotService::LIFECYCLE_EVANGELIST => $this->t('Evangelist'),
        HubSpotService::LIFECYCLE_OTHER => $this->t('Other'),
        HubSpotService::VALUE_TOKEN => $this->t('By Token'),
      ],
      '#default_value' => $this->configuration['lifecyclestage'],
    ];

    $form['lifecyclestage_token'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Lifecycle Stage Token'),
      '#description' => $this->t('Token value that evaluates to a lifecycle stage (e.g., "lead", "customer", "marketingqualifiedlead").'),
      '#default_value' => $this->configuration['lifecyclestage_token'],
      '#states' => [
        'visible' => [
          ':input[name="lifecyclestage"]' => ['value' => HubSpotService::VALUE_TOKEN],
        ],
        'required' => [
          ':input[name="lifecyclestage"]' => ['value' => HubSpotService::VALUE_TOKEN],
        ],
      ],
      '#eca_token_replacement' => TRUE,
    ];

    $form = $this->addYamlPropertiesField($form, $form_state, 'contact');
    $form = $this->addTokenOutputField($form, $form_state, $this->t('Provide the name of a token that holds the contact data including ID, email, and all properties.'));

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    // Build contact properties.
    $properties = [];

    // Add properties if provided.
    $email = $this->tokenService->replacePlain($this->configuration['email']);
    if (!empty($email)) {
      $properties['email'] = $email;
    }

    $firstname = $this->tokenService->replaceClear($this->configuration['firstname']);
    if (!empty($firstname)) {
      $properties['firstname'] = $firstname;
    }

    $lastname = $this->tokenService->replaceClear($this->configuration['lastname']);
    if (!empty($lastname)) {
      $properties['lastname'] = $lastname;
    }

    $phone = $this->tokenService->replaceClear($this->configuration['phone']);
    if (!empty($phone)) {
      $properties['phone'] = $phone;
    }

    $company = $this->tokenService->replaceClear($this->configuration['company']);
    if (!empty($company)) {
      $properties['company'] = $company;
    }

    // Handle lifecycle stage - check if using token or direct value.
    $lifecyclestage = $this->configuration['lifecyclestage'] ?? '';
    if ($lifecyclestage === HubSpotService::VALUE_TOKEN) {
      $token_value = strtolower(trim($this->tokenService->replacePlain($this->configuration['lifecyclestage_token'] ?? '')));
      if (!empty($token_value)) {
        $properties['lifecyclestage'] = $token_value;
      }
    }
    elseif (!empty($lifecyclestage)) {
      $properties['lifecyclestage'] = $lifecyclestage;
    }

    // Merge additional properties from YAML.
    $properties = $this->mergeYamlSettings($properties, 'additional_properties');

    // Create new contact.
    $response = $this->hubspotService->createContact($properties);

    // Store response in token if configured.
    $this->storeResponseToken($response);
  }

}
