<?php

declare(strict_types=1);

namespace Drupal\eca_hubspot\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;

/**
 * Create a new HubSpot lead.
 */
#[Action(
  id: 'eca_hubspot_create_lead',
  label: new TranslatableMarkup('HubSpot: Create Lead'),
  category: new TranslatableMarkup('HubSpot'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Create a new lead record in HubSpot. Leads must be associated with a contact.'),
  version_introduced: '1.0.0',
)]
final class CreateLead extends HubSpotActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'lead_name' => '',
      'lead_type' => '',
      'lead_label' => '',
      'associated_contact_id' => '',
      'associated_company_ids' => '',
      'additional_properties' => '',
      'token_name' => '',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['lead_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Lead Name'),
      '#description' => $this->t('The name of the lead (required).'),
      '#default_value' => $this->configuration['lead_name'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['lead_type'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Lead Type'),
      '#description' => $this->t('The type of lead (e.g., "NEW_BUSINESS", "EXISTING_BUSINESS").'),
      '#default_value' => $this->configuration['lead_type'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['lead_label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Lead Label'),
      '#description' => $this->t('Custom label for the lead.'),
      '#default_value' => $this->configuration['lead_label'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['associated_contact_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Associated Contact ID'),
      '#description' => $this->t('The contact ID to associate with this lead (required).'),
      '#default_value' => $this->configuration['associated_contact_id'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['associated_company_ids'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Associated Company IDs'),
      '#description' => $this->t('Comma-separated list of company IDs to associate with this lead.'),
      '#default_value' => $this->configuration['associated_company_ids'],
      '#eca_token_replacement' => TRUE,
    ];

    $form = $this->addYamlPropertiesField($form, $form_state, 'lead');
    $form = $this->addTokenOutputField($form, $form_state, $this->t('Provide the name of a token that holds the created lead data including ID and all properties.'));

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    // Build lead properties.
    $properties = [];

    // Required: Lead name.
    $lead_name = $this->tokenService->replaceClear($this->configuration['lead_name']);
    if (empty($lead_name)) {
      $this->logError('Lead name is required to create lead.');
      return;
    }
    $properties['hs_lead_name'] = $lead_name;

    // Optional properties.
    $lead_type = $this->tokenService->replacePlain($this->configuration['lead_type']);
    if (!empty($lead_type)) {
      $properties['hs_lead_type'] = $lead_type;
    }

    $lead_label = $this->tokenService->replacePlain($this->configuration['lead_label']);
    if (!empty($lead_label)) {
      $properties['hs_lead_label'] = $lead_label;
    }

    // Merge additional properties from YAML.
    $properties = $this->mergeYamlSettings($properties, 'additional_properties');

    // Build associations array - contact is required.
    $associations = [];

    $contact_id = $this->tokenService->replacePlain($this->configuration['associated_contact_id']);
    if (empty($contact_id)) {
      $this->logError('Associated contact ID is required to create lead.');
      return;
    }
    $associations['contact'] = [$contact_id];

    $company_ids = $this->tokenService->replacePlain($this->configuration['associated_company_ids']);
    if (!empty($company_ids)) {
      $associations['company'] = array_map('trim', explode(',', $company_ids));
    }

    // Create new lead.
    $response = $this->hubspotService->createLead($properties, $associations);

    // Store response in token if configured.
    $this->storeResponseToken($response);
  }

}
