<?php

declare(strict_types=1);

namespace Drupal\eca_hubspot\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;

/**
 * Delete a HubSpot company.
 */
#[Action(
  id: 'eca_hubspot_delete_company',
  label: new TranslatableMarkup('HubSpot: Delete Company'),
  category: new TranslatableMarkup('HubSpot'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Delete a company record from HubSpot by ID.'),
  version_introduced: '1.0.0',
)]
final class DeleteCompany extends HubSpotActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'company_id' => '',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['company_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Company ID'),
      '#description' => $this->t('The HubSpot company ID to delete.'),
      '#default_value' => $this->configuration['company_id'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $company_id = $this->tokenService->replacePlain($this->configuration['company_id']);
    if (empty($company_id)) {
      $this->logError('Company ID is required to delete company.');
      return;
    }

    // Delete the company.
    $success = $this->hubspotService->deleteCompany($company_id);

    if (!$success) {
      $this->logError('Failed to delete company with ID: @id', ['@id' => $company_id]);
    }
  }

}
