<?php

declare(strict_types=1);

namespace Drupal\eca_hubspot\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;

/**
 * Delete a HubSpot note.
 */
#[Action(
  id: 'eca_hubspot_delete_note',
  label: new TranslatableMarkup('HubSpot: Delete Note'),
  category: new TranslatableMarkup('HubSpot'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Delete a note record from HubSpot by ID.'),
  version_introduced: '1.0.0',
)]
final class DeleteNote extends HubSpotActionBase {

  public function defaultConfiguration(): array {
    return [
      'note_id' => '',
    ] + parent::defaultConfiguration();
  }

  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['note_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Note ID'),
      '#description' => $this->t('The HubSpot note ID to delete.'),
      '#default_value' => $this->configuration['note_id'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    return $form;
  }

  public function execute(): void {
    $note_id = $this->tokenService->replacePlain($this->configuration['note_id']);
    if (empty($note_id)) {
      $this->logError('Note ID is required to delete note.');
      return;
    }

    $success = $this->hubspotService->deleteNote($note_id);

    if (!$success) {
      $this->logError('Failed to delete note with ID: @id', ['@id' => $note_id]);
    }
  }

}
