<?php

declare(strict_types=1);

namespace Drupal\eca_hubspot\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;

/**
 * Delete a HubSpot task.
 */
#[Action(
  id: 'eca_hubspot_delete_task',
  label: new TranslatableMarkup('HubSpot: Delete Task'),
  category: new TranslatableMarkup('HubSpot'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Delete a task record from HubSpot by ID.'),
  version_introduced: '1.0.0',
)]
final class DeleteTask extends HubSpotActionBase {

  public function defaultConfiguration(): array {
    return [
      'task_id' => '',
    ] + parent::defaultConfiguration();
  }

  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['task_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Task ID'),
      '#description' => $this->t('The HubSpot task ID to delete.'),
      '#default_value' => $this->configuration['task_id'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    return $form;
  }

  public function execute(): void {
    $task_id = $this->tokenService->replacePlain($this->configuration['task_id']);
    if (empty($task_id)) {
      $this->logError('Task ID is required to delete task.');
      return;
    }

    $success = $this->hubspotService->deleteTask($task_id);

    if (!$success) {
      $this->logError('Failed to delete task with ID: @id', ['@id' => $task_id]);
    }
  }

}
