<?php

declare(strict_types=1);

namespace Drupal\eca_hubspot\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;

/**
 * Get a HubSpot contact.
 */
#[Action(
  id: 'eca_hubspot_get_contact',
  label: new TranslatableMarkup('HubSpot: Get Contact'),
  category: new TranslatableMarkup('HubSpot'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Retrieve contact by ID or email for workflow decisions.'),
  version_introduced: '1.0.0',
)]
final class GetContact extends HubSpotActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'contact_id_or_email' => '',
      'properties' => '',
      'token_name' => '',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['contact_id_or_email'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Contact ID or Email'),
      '#description' => $this->t('The HubSpot contact ID or email address to retrieve.'),
      '#default_value' => $this->configuration['contact_id_or_email'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['properties'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Properties to Retrieve'),
      '#description' => $this->t('Comma-separated list of property names to retrieve. Leave empty to retrieve the default properties. Example: firstname,lastname,email,phone'),
      '#default_value' => $this->configuration['properties'],
      '#eca_token_replacement' => TRUE,
    ];

    $form = $this->addTokenOutputField($form, $form_state, $this->t('Provide the name of a token that holds the contact data including all requested properties.'));

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $contact_id_or_email = $this->tokenService->replacePlain($this->configuration['contact_id_or_email']);
    if (empty($contact_id_or_email)) {
      $this->logError('Contact ID or Email is required to get contact.');
      return;
    }

    // Parse properties list.
    $properties = [];
    $properties_string = $this->tokenService->replacePlain($this->configuration['properties']);
    if (!empty($properties_string)) {
      $properties = array_map('trim', explode(',', $properties_string));
    }

    // Get the contact.
    $response = $this->hubspotService->getContact($contact_id_or_email, $properties);

    // Store response in token if configured.
    $this->storeResponseToken($response);
  }

}
