<?php

declare(strict_types=1);

namespace Drupal\eca_hubspot\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;

/**
 * Get a HubSpot lead.
 */
#[Action(
  id: 'eca_hubspot_get_lead',
  label: new TranslatableMarkup('HubSpot: Get Lead'),
  category: new TranslatableMarkup('HubSpot'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Retrieve a lead record from HubSpot by ID.'),
  version_introduced: '1.0.0',
)]
final class GetLead extends HubSpotActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'lead_id' => '',
      'properties' => '',
      'include_associations' => FALSE,
      'token_name' => '',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['lead_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Lead ID'),
      '#description' => $this->t('The HubSpot lead ID to retrieve.'),
      '#default_value' => $this->configuration['lead_id'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['properties'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Properties to Retrieve'),
      '#description' => $this->t('Comma-separated list of property names to retrieve. Leave empty to retrieve all properties.'),
      '#default_value' => $this->configuration['properties'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['include_associations'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Include Associations'),
      '#description' => $this->t('Include associated contacts and companies in the response.'),
      '#default_value' => $this->configuration['include_associations'],
    ];

    $form = $this->addTokenOutputField($form, $form_state, $this->t('Provide the name of a token that holds the retrieved lead data including ID, properties, and optionally associations.'));

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $lead_id = $this->tokenService->replacePlain($this->configuration['lead_id']);
    if (empty($lead_id)) {
      $this->logError('Lead ID is required to get lead.');
      return;
    }

    // Parse properties list.
    $properties = [];
    $properties_list = $this->tokenService->replacePlain($this->configuration['properties']);
    if (!empty($properties_list)) {
      $properties = array_map('trim', explode(',', $properties_list));
    }

    $include_associations = !empty($this->configuration['include_associations']);

    // Get the lead.
    $response = $this->hubspotService->getLead($lead_id, $properties, $include_associations);

    // Store response in token if configured.
    $this->storeResponseToken($response);
  }

}
