<?php

declare(strict_types=1);

namespace Drupal\eca_hubspot\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;

/**
 * Get a HubSpot ticket.
 */
#[Action(
  id: 'eca_hubspot_get_ticket',
  label: new TranslatableMarkup('HubSpot: Get Ticket'),
  category: new TranslatableMarkup('HubSpot'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Retrieve a ticket record from HubSpot by ID.'),
  version_introduced: '1.0.0',
)]
final class GetTicket extends HubSpotActionBase {

  public function defaultConfiguration(): array {
    return [
      'ticket_id' => '',
      'properties' => '',
      'include_associations' => FALSE,
      'token_name' => '',
    ] + parent::defaultConfiguration();
  }

  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['ticket_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Ticket ID'),
      '#description' => $this->t('The HubSpot ticket ID to retrieve.'),
      '#default_value' => $this->configuration['ticket_id'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['properties'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Properties to Retrieve'),
      '#description' => $this->t('Comma-separated list of property names. Leave empty for all.'),
      '#default_value' => $this->configuration['properties'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['include_associations'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Include Associations'),
      '#description' => $this->t('Include associated contacts, companies, and deals.'),
      '#default_value' => $this->configuration['include_associations'],
    ];

    $form = $this->addTokenOutputField($form, $form_state, $this->t('Token that holds the retrieved ticket data.'));

    return $form;
  }

  public function execute(): void {
    $ticket_id = $this->tokenService->replacePlain($this->configuration['ticket_id']);
    if (empty($ticket_id)) {
      $this->logError('Ticket ID is required to get ticket.');
      return;
    }

    $properties = [];
    $properties_list = $this->tokenService->replacePlain($this->configuration['properties']);
    if (!empty($properties_list)) {
      $properties = array_map('trim', explode(',', $properties_list));
    }

    $include_associations = !empty($this->configuration['include_associations']);

    $response = $this->hubspotService->getTicket($ticket_id, $properties, $include_associations);
    $this->storeResponseToken($response);
  }

}
