<?php

declare(strict_types=1);

namespace Drupal\eca_hubspot\Plugin\Action;

use Drupal\Core\Form\FormStateInterface;
use Drupal\eca\Plugin\Action\ConfigurableActionBase;
use Drupal\eca_hubspot\Service\HubSpotService;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Yaml\Yaml;

/**
 * Base class for HubSpot ECA actions.
 */
abstract class HubSpotActionBase extends ConfigurableActionBase {

  /**
   * The HubSpot service.
   */
  protected HubSpotService $hubspotService;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->hubspotService = $container->get('eca_hubspot.hubspot');
    return $instance;
  }

  /**
   * Gets the HubSpot service instance.
   *
   * @return \Drupal\eca_hubspot\Service\HubSpotService
   *   The HubSpot service.
   */
  protected function getHubSpotService(): HubSpotService {
    return $this->hubspotService;
  }

  /**
   * Gets an object ID from configuration with token replacement.
   *
   * @param string $config_key
   *   The configuration key containing the object ID.
   *
   * @return string
   *   The object ID.
   */
  protected function getObjectId(string $config_key = 'object_id'): string {
    return $this->tokenService->replacePlain($this->configuration[$config_key]);
  }

  /**
   * Parses YAML settings and merges them with base data.
   *
   * @param array $base_data
   *   The base data array to merge into.
   * @param string $config_key
   *   The configuration key containing the YAML string.
   *
   * @return array
   *   The merged data array.
   */
  protected function mergeYamlSettings(array $base_data, string $config_key): array {
    $yaml_string = $this->tokenService->replacePlain($this->configuration[$config_key] ?? '');

    if (empty($yaml_string)) {
      return $base_data;
    }

    try {
      $yaml_data = Yaml::parse($yaml_string);
      if (is_array($yaml_data)) {
        return $this->mergeArraysRecursive($base_data, $yaml_data);
      }
    }
    catch (\Exception $e) {
      // Log YAML parsing error but continue with basic settings.
      if (method_exists($this, 'logger') && isset($this->logger)) {
        $this->logger->error('Failed to parse YAML settings from @key: @message', [
          '@key' => $config_key,
          '@message' => $e->getMessage(),
        ]);
      }
    }

    return $base_data;
  }

  /**
   * Recursively merge arrays, preserving nested structures.
   *
   * @param array $array1
   *   The base array.
   * @param array $array2
   *   The array to merge in.
   *
   * @return array
   *   The merged array.
   */
  protected function mergeArraysRecursive(array $array1, array $array2): array {
    foreach ($array2 as $key => $value) {
      if (is_array($value) && isset($array1[$key]) && is_array($array1[$key])) {
        $array1[$key] = $this->mergeArraysRecursive($array1[$key], $value);
      }
      else {
        $array1[$key] = $value;
      }
    }
    return $array1;
  }

  /**
   * Adds YAML properties field for additional/custom properties.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   * @param string $object_type
   *   The HubSpot object type (for documentation link).
   *
   * @return array
   *   The form array with the field added.
   */
  protected function addYamlPropertiesField(array $form, FormStateInterface $form_state, string $object_type): array {
    $urls = [
      'contact' => 'https://developers.hubspot.com/docs/api/crm/contacts',
      'company' => 'https://developers.hubspot.com/docs/api/crm/companies',
      'deal' => 'https://developers.hubspot.com/docs/api/crm/deals',
      'ticket' => 'https://developers.hubspot.com/docs/api/crm/tickets',
    ];

    $url = $urls[$object_type] ?? 'https://developers.hubspot.com/docs/api/overview';

    $form['additional_properties'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Additional Properties (YAML)'),
      '#description' => $this->t('Additional or custom properties in YAML format. See <a href="@url" target="_blank">HubSpot API documentation</a> for available properties. Example:<br><code>website: "https://example.com"<br>numberofemployees: 50<br>custom_property: "value"</code>', [
        '@url' => $url,
      ]),
      '#default_value' => $this->configuration['additional_properties'] ?? '',
      '#rows' => 8,
      '#eca_token_replacement' => TRUE,
    ];

    return $form;
  }

  /**
   * Adds token output field for storing response data.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   * @param string|\Drupal\Core\StringTranslation\TranslatableMarkup $description
   *   Optional custom description.
   *
   * @return array
   *   The form array with the field added.
   */
  protected function addTokenOutputField(array $form, FormStateInterface $form_state, string|\Drupal\Core\StringTranslation\TranslatableMarkup $description = ''): array {
    if (empty($description)) {
      $description = $this->t('Provide the name of a token that holds the response data from HubSpot.');
    }

    $form['token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Token name'),
      '#description' => $description,
      '#default_value' => $this->configuration['token_name'] ?? '',
      '#eca_token_replacement' => TRUE,
      '#weight' => 50,
    ];

    return $form;
  }

  /**
   * Stores response data in a token if configured.
   *
   * @param array|null $response
   *   The response data to store.
   */
  protected function storeResponseToken(?array $response): void {
    if ($response && !empty($this->configuration['token_name'])) {
      $token_name = $this->tokenService->replacePlain($this->configuration['token_name']);
      $this->tokenService->addTokenData($token_name, $response);
    }
  }

  /**
   * Logs an error message.
   *
   * @param string $message
   *   The error message.
   * @param array $context
   *   The message context variables.
   */
  protected function logError(string $message, array $context = []): void {
    if (method_exists($this, 'logger') && isset($this->logger)) {
      $this->logger->error($message, $context);
    }
  }

}
