<?php

declare(strict_types=1);

namespace Drupal\eca_hubspot\Plugin\Action;

use Drupal\Core\Form\FormStateInterface;

/**
 * Base class for HubSpot search actions.
 */
abstract class HubSpotSearchActionBase extends HubSpotActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'filters' => '',
      'sort_by' => '',
      'limit' => '100',
      'token_name' => '',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    $object_type = $this->getObjectType();
    $object_label = $this->getObjectLabel();

    $form['filters'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Search Filters (YAML)'),
      '#description' => $this->t('Search filters in YAML format. Each filter should specify propertyName, operator, and value. Common operators: EQ (equals), NEQ (not equals), LT/LTE/GT/GTE (comparisons), CONTAINS_TOKEN (partial match with wildcards), HAS_PROPERTY, NOT_HAS_PROPERTY, IN, NOT_IN. See <a href="@url" target="_blank">HubSpot Search API documentation</a> for full list.<br><br>Example using inline format (recommended - no indentation required):<br><code>@example</code><br><br>Alternative block format (requires 2-space indentation):<br><code>@block_example</code>', [
        '@url' => 'https://developers.hubspot.com/docs/api/crm/search',
        '@example' => $this->getFilterExample(),
        '@block_example' => $this->getBlockFilterExample(),
      ]),
      '#default_value' => $this->configuration['filters'],
      '#rows' => 8,
      '#eca_token_replacement' => TRUE,
    ];

    $form['sort_by'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Sort By'),
      '#description' => $this->t('Property name to sort results by (e.g., "createdate"). Leave empty for default sorting.'),
      '#default_value' => $this->configuration['sort_by'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['limit'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Limit'),
      '#description' => $this->t('Maximum number of results to return (1-100). Default is 100.'),
      '#default_value' => $this->configuration['limit'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form = $this->addTokenOutputField($form, $form_state, $this->t('Provide the name of a token that holds the search results array including total count and matching @objects.', [
      '@objects' => $object_label,
    ]));

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    // Parse filters from YAML.
    $filters = [];
    $filters_yaml = $this->tokenService->replacePlain($this->configuration['filters']);
    if (!empty($filters_yaml)) {
      $filters = $this->mergeYamlSettings([], 'filters');
      // If mergeYamlSettings returns empty, try direct parsing.
      if (empty($filters)) {
        try {
          $filters = \Symfony\Component\Yaml\Yaml::parse($filters_yaml);
          if (!is_array($filters)) {
            $filters = [];
          }
        }
        catch (\Exception $e) {
          $this->logError('Failed to parse search filters YAML: @message', [
            '@message' => $e->getMessage(),
          ]);
          return;
        }
      }
    }

    // Get sort by property.
    $sort_by = $this->tokenService->replacePlain($this->configuration['sort_by']);

    // Get limit.
    $limit = (int) $this->tokenService->replacePlain($this->configuration['limit']);
    if ($limit < 1 || $limit > 100) {
      $limit = 100;
    }

    // Perform search using the service method.
    $response = $this->performSearch($filters, $sort_by, $limit);

    // Store response in token if configured.
    $this->storeResponseToken($response);
  }

  /**
   * Gets the object type being searched.
   *
   * @return string
   *   The object type (e.g., 'contacts', 'companies', 'deals').
   */
  abstract protected function getObjectType(): string;

  /**
   * Gets the human-readable object label (plural form).
   *
   * @return string
   *   The object label (e.g., 'contacts', 'companies', 'deals').
   */
  abstract protected function getObjectLabel(): string;

  /**
   * Gets an example filter for the form description.
   *
   * @return string
   *   Example filter in inline YAML format.
   */
  abstract protected function getFilterExample(): string;

  /**
   * Gets a block format example filter for the form description.
   *
   * @return string
   *   Example filter in block YAML format.
   */
  abstract protected function getBlockFilterExample(): string;

  /**
   * Performs the search operation.
   *
   * @param array $filters
   *   The search filters.
   * @param string $sort_by
   *   The property to sort by.
   * @param int $limit
   *   The maximum number of results.
   *
   * @return array|null
   *   The search results or NULL on failure.
   */
  abstract protected function performSearch(array $filters, string $sort_by, int $limit): ?array;

}
