<?php

declare(strict_types=1);

namespace Drupal\eca_hubspot\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;

/**
 * List HubSpot pipelines.
 */
#[Action(
  id: 'eca_hubspot_list_pipelines',
  label: new TranslatableMarkup('HubSpot: List Pipelines'),
  category: new TranslatableMarkup('HubSpot'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('List all pipelines and their stages for an object type (deals, tickets, leads, etc.).'),
  version_introduced: '1.0.0',
)]
final class ListPipelines extends HubSpotActionBase {

  public function defaultConfiguration(): array {
    return [
      'object_type' => 'deals',
      'object_type_custom' => '',
      'token_name' => '',
    ] + parent::defaultConfiguration();
  }

  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['object_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Object Type'),
      '#description' => $this->t('The type of object to list pipelines for.'),
      '#options' => [
        'deals' => $this->t('Deals'),
        'tickets' => $this->t('Tickets'),
        'leads' => $this->t('Leads'),
        'appointments' => $this->t('Appointments'),
        'courses' => $this->t('Courses'),
        'listings' => $this->t('Listings'),
        'orders' => $this->t('Orders'),
        'services' => $this->t('Services'),
      ],
      '#default_value' => $this->configuration['object_type'],
      '#required' => TRUE,
    ];

    $form['object_type_custom'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Custom Object Type'),
      '#description' => $this->t('If using a custom object type, enter it here. This will override the selection above. Use token replacement for dynamic values.'),
      '#default_value' => $this->configuration['object_type_custom'] ?? '',
      '#eca_token_replacement' => TRUE,
    ];

    $form = $this->addTokenOutputField($form, $form_state, $this->t('Token that holds the array of pipelines with their stages. Each pipeline includes: id, label, displayOrder, and stages array (each stage has id, label, displayOrder, metadata).'));

    return $form;
  }

  public function execute(): void {
    // Check if custom object type is provided.
    $custom_type = $this->tokenService->replacePlain($this->configuration['object_type_custom'] ?? '');
    $object_type = !empty($custom_type) ? $custom_type : $this->configuration['object_type'];

    if (empty($object_type)) {
      $this->logError('Object type is required to list pipelines.');
      return;
    }

    $response = $this->hubspotService->listPipelines($object_type);
    $this->storeResponseToken($response);
  }

}
